<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ImportCommodityCodes extends Command
{
    protected $signature = 'commodity-codes:import {file? : Path to CSV file}';
    protected $description = 'Import commodity codes from CSV file';

    public function handle()
    {
        $file = $this->argument('file') ?? storage_path('app/temp/codes.csv');
        
        if (!file_exists($file)) {
            $this->error("File not found: {$file}");
            return 1;
        }

        $this->info("Importing commodity codes from: {$file}");

        // Read CSV
        $handle = fopen($file, 'r');
        $header = fgetcsv($handle);
        
        $this->info("Columns: " . implode(', ', $header));

        // Clear existing data
        $existingCount = DB::table('commodity_codes')->count();
        if ($existingCount > 0) {
            if ($this->confirm("This will replace {$existingCount} existing commodity codes. Continue?", true)) {
                DB::table('commodity_codes')->truncate();
                $this->info("Cleared existing commodity codes.");
            } else {
                $this->info("Aborted.");
                fclose($handle);
                return 0;
            }
        }

        $count = 0;
        $now = now();

        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) < 6) continue;

            DB::table('commodity_codes')->insert([
                'id' => $row[0],           // id (commodity code number)
                'order' => (int) $row[1],  // order
                'Gender' => $row[2] ?: null,       // Gender
                'Type' => $row[3] ?: null,         // Type
                'Composition' => $row[4] ?: null,  // Composition
                'description' => $row[5],  // description
                'created_at' => $now,
                'updated_at' => $now,
            ]);
            $count++;
        }

        fclose($handle);

        $this->info("✓ Imported {$count} commodity codes successfully.");
        
        return 0;
    }
}


































