<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class FixSizeTypos extends Command
{
    protected $signature = 'sizes:fix-typos 
                            {--dry-run : Show what would be changed without making changes}';

    protected $description = 'Fix size typos: convert 06 to 6 and 08 to 8';

    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        $this->info('Fixing size typos: 06 → 6, 08 → 8');
        $this->newLine();
        
        // Size mappings
        $mappings = [
            ['old_id' => 1052, 'old_name' => '06', 'new_id' => 7, 'new_name' => '6'],
            ['old_id' => 1053, 'old_name' => '08', 'new_id' => 8, 'new_name' => '8'],
        ];
        
        // Tables to update
        $tables = [
            'customer_order_line_quantities' => 'sizes_id',
            'shipment_line_sizes' => 'sizes_id',
            'prices' => 'sizes_id',
            'samples' => 'sizes_id',
        ];
        
        $totalUpdates = 0;
        
        // First, show what will be updated
        $this->info('Analyzing records to update:');
        $this->newLine();
        
        foreach ($mappings as $mapping) {
            $this->line("Size {$mapping['old_name']} (ID: {$mapping['old_id']}) → {$mapping['new_name']} (ID: {$mapping['new_id']})");
            
            foreach ($tables as $table => $column) {
                $count = DB::table($table)
                    ->where($column, $mapping['old_id'])
                    ->whereNull('deleted_at')
                    ->count();
                
                if ($count > 0) {
                    $this->line("  - {$table}: {$count} records");
                    $totalUpdates += $count;
                }
            }
            $this->newLine();
        }
        
        if ($totalUpdates === 0) {
            $this->info('No records found to update!');
            return 0;
        }
        
        $this->info("Total records to update: {$totalUpdates}");
        $this->newLine();
        
        if ($dryRun) {
            $this->warn('DRY RUN MODE - No changes will be made');
            return 0;
        }
        
        if (!$this->confirm('Do you want to proceed with these updates?', true)) {
            $this->info('Operation cancelled.');
            return 0;
        }
        
        // Perform updates in a transaction
        DB::beginTransaction();
        
        try {
            $updated = 0;
            
            foreach ($mappings as $mapping) {
                $this->info("Processing size {$mapping['old_name']} → {$mapping['new_name']}...");
                
                foreach ($tables as $table => $column) {
                    $count = DB::table($table)
                        ->where($column, $mapping['old_id'])
                        ->whereNull('deleted_at')
                        ->update([
                            $column => $mapping['new_id'],
                            'updated_at' => now(),
                        ]);
                    
                    if ($count > 0) {
                        $this->line("  ✓ Updated {$count} records in {$table}");
                        $updated += $count;
                    }
                }
            }
            
            // Soft delete the old size records
            $this->info('Soft deleting old size records...');
            DB::table('sizes')
                ->whereIn('id', [1052, 1053])
                ->update([
                    'deleted_at' => now(),
                    'updated_at' => now(),
                ]);
            $this->line('  ✓ Soft deleted sizes 06 and 08');
            
            DB::commit();
            
            $this->newLine();
            $this->info("✅ Successfully updated {$updated} records!");
            $this->info('✅ Soft deleted old size records 06 and 08');
            
            // Invalidate cache for affected shipment lines
            $this->newLine();
            $this->info('Invalidating totals cache for affected shipment lines...');
            
            $affectedShipmentLines = DB::table('shipment_line_sizes')
                ->whereIn('sizes_id', [7, 8])
                ->whereNull('deleted_at')
                ->distinct()
                ->pluck('shipment_line_id');
            
            if ($affectedShipmentLines->isNotEmpty()) {
                DB::table('total_cache')
                    ->where('entity_type', 'shipment_line')
                    ->whereIn('entity_id', $affectedShipmentLines)
                    ->update(['fresh_at' => null]);
                
                $this->info("  ✓ Invalidated cache for {$affectedShipmentLines->count()} shipment lines");
                $this->line('  (Cache will be automatically refreshed on next access)');
            }
            
            return 0;
            
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('Error occurred: ' . $e->getMessage());
            $this->error('All changes have been rolled back.');
            return 1;
        }
    }
}

