<?php

namespace App\Console\Commands;

use App\Models\ShipmentLine;
use App\Models\TotalCache;
use App\Services\TotalCacheService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class FixShipmentLineTotalsCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'shipment-lines:fix-totals-cache 
                            {--list-only : Only list the affected shipment lines without updating}
                            {--limit= : Limit the number of records to process}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Find and fix shipment lines with zero or missing totals cache';

    protected TotalCacheService $totalCacheService;

    /**
     * Execute the console command.
     */
    public function handle(TotalCacheService $totalCacheService)
    {
        $this->totalCacheService = $totalCacheService;
        
        $this->info('Finding shipment lines with zero or missing totals cache...');
        
        // Get all shipment lines with their cache data
        $query = $this->getShipmentLinesQuery();
        
        if ($limit = $this->option('limit')) {
            $query->limit((int) $limit);
        }
        
        $affectedLines = $query->get();
        
        if ($affectedLines->isEmpty()) {
            $this->info('No shipment lines found with zero or missing totals cache.');
            return 0;
        }
        
        $this->info("Found {$affectedLines->count()} shipment lines with issues.");
        $this->newLine();
        
        // Display the list
        $this->displayAffectedLines($affectedLines);
        
        if ($this->option('list-only')) {
            $this->info('List-only mode: No updates performed.');
            return 0;
        }
        
        // Ask for confirmation
        if (!$this->confirm('Do you want to update the totals cache for these lines?', true)) {
            $this->info('Operation cancelled.');
            return 0;
        }
        
        // Process updates
        $this->newLine();
        $this->info('Updating totals cache...');
        
        $progressBar = $this->output->createProgressBar($affectedLines->count());
        $progressBar->start();
        
        $updated = 0;
        $skipped = 0;
        $errors = 0;
        
        foreach ($affectedLines as $line) {
            try {
                // Load the shipment line with necessary relationships
                $shipmentLine = ShipmentLine::with([
                    'customer_order_lines.customer_order_line_quantities.sizes',
                    'shipment_line_sizes'
                ])->find($line->id);
                
                if (!$shipmentLine) {
                    $skipped++;
                    $progressBar->advance();
                    continue;
                }
                
                // Check if price data is available
                if ($this->hasPriceData($shipmentLine)) {
                    // Warm up the cache
                    $this->totalCacheService->warmupShipmentLine($shipmentLine);
                    $updated++;
                } else {
                    $skipped++;
                }
                
                $progressBar->advance();
                
            } catch (\Exception $e) {
                $errors++;
                $this->error("\nError processing shipment line {$line->id}: " . $e->getMessage());
                $progressBar->advance();
            }
        }
        
        $progressBar->finish();
        $this->newLine(2);
        
        // Display summary
        $this->info("Summary:");
        $this->table(
            ['Status', 'Count'],
            [
                ['Updated', $updated],
                ['Skipped (no price data)', $skipped],
                ['Errors', $errors],
                ['Total', $affectedLines->count()],
            ]
        );
        
        return 0;
    }
    
    /**
     * Get query for shipment lines with zero or missing totals cache.
     */
    protected function getShipmentLinesQuery()
    {
        return DB::table('shipment_lines')
            ->select(
                'shipment_lines.id',
                'shipment_lines.shipment_id',
                'shipment_lines.customer_order_lines_id',
                'total_cache.cached_data',
                'total_cache.fresh_at'
            )
            ->leftJoin('total_cache', function ($join) {
                $join->on('total_cache.entity_id', '=', 'shipment_lines.id')
                     ->where('total_cache.entity_type', '=', 'shipment_line')
                     ->where('total_cache.cache_key', '=', 'prices');
            })
            ->whereNull('shipment_lines.deleted_at')
            ->where(function ($query) {
                // Either no cache entry exists
                $query->whereNull('total_cache.id')
                    // Or cache exists but has zero subtotal_base
                    ->orWhereRaw("JSON_EXTRACT(total_cache.cached_data, '$.subtotal_base') = 0")
                    // Or cache exists but subtotal_base is null
                    ->orWhereRaw("JSON_EXTRACT(total_cache.cached_data, '$.subtotal_base') IS NULL");
            })
            ->orderBy('shipment_lines.id');
    }
    
    /**
     * Display affected lines in a table.
     */
    protected function displayAffectedLines($lines)
    {
        $tableData = [];
        
        foreach ($lines->take(20) as $line) {
            $cacheStatus = $line->cached_data ? 'Exists (zero value)' : 'Missing';
            $subtotalBase = $line->cached_data ? 
                (json_decode($line->cached_data, true)['subtotal_base'] ?? 'null') : 
                'N/A';
            
            $tableData[] = [
                $line->id,
                $line->shipment_id ?? 'N/A',
                $line->customer_order_lines_id,
                $cacheStatus,
                $subtotalBase,
            ];
        }
        
        $this->table(
            ['Shipment Line ID', 'Shipment ID', 'Order Line ID', 'Cache Status', 'Subtotal Base'],
            $tableData
        );
        
        if ($lines->count() > 20) {
            $this->info("... and " . ($lines->count() - 20) . " more lines.");
        }
    }
    
    /**
     * Check if shipment line has price data available.
     */
    protected function hasPriceData(ShipmentLine $shipmentLine): bool
    {
        try {
            // Check if customer order line has quantities with price models
            $orderLine = $shipmentLine->customer_order_lines;
            
            if (!$orderLine) {
                return false;
            }
            
            // Check if any quantity has a price model
            foreach ($orderLine->customer_order_line_quantities as $qty) {
                $priceModel = $qty->price_model;
                if (!empty($priceModel) && isset($priceModel['subtotal_base'])) {
                    return true;
                }
            }
            
            return false;
        } catch (\Exception $e) {
            return false;
        }
    }
}

