<?php

namespace App\Console\Commands;

use App\Models\Price;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class ConfirmAsosPricesToday extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'asos:confirm-prices-today';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Mark all ASOS prices created or updated today as confirmed';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Finding ASOS prices created/updated today...');
        
        $today = Carbon::today();
        
        // Find all prices for ASOS customer (ID: 6) created or updated today
        // We need to join through style_versions -> styles to find ASOS prices
        $prices = Price::whereHas('style_versions.styles', function($query) {
            $query->where('customers_id', 6); // ASOS customer ID
        })
        ->where(function($query) use ($today) {
            $query->whereDate('created_at', $today)
                  ->orWhereDate('updated_at', $today);
        })
        ->where(function($query) {
            // Only update prices that aren't already confirmed
            $query->where('cmt_status', '!=', 'confirmed')
                  ->orWhere('quote_status', '!=', 'confirmed');
        })
        ->get();
        
        if ($prices->isEmpty()) {
            $this->info('No ASOS prices found that need to be confirmed.');
            return 0;
        }
        
        $this->info("Found {$prices->count()} price(s) to confirm.");
        
        $bar = $this->output->createProgressBar($prices->count());
        $bar->start();
        
        $confirmed = 0;
        
        foreach ($prices as $price) {
            // Update using the model to trigger observers and cache updates
            $price->update([
                'cmt_status' => 'confirmed',
                'quote_status' => 'confirmed',
            ]);
            
            $confirmed++;
            $bar->advance();
        }
        
        $bar->finish();
        $this->newLine(2);
        
        $this->info("✓ Successfully confirmed {$confirmed} ASOS price(s).");
        $this->info('Price caches and totals will be updated automatically.');
        
        return 0;
    }
}
