<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CommissionImportBatchFile;
use App\Jobs\ProcessCommissionFile;
use Illuminate\Support\Facades\Log;

class CommissionDrainPending extends Command
{
    protected $signature = 'commission:drain {--limit=5} {--batch=}';
    protected $description = 'Synchronously process a few pending/processing commission files (fallback when queue is slow).';

    public function handle(): int
    {
        $limit = (int)$this->option('limit');
        $batchId = $this->option('batch');

        $query = CommissionImportBatchFile::query()
            ->whereIn('status', ['pending', 'processing'])
            ->orderBy('id');

        if (!empty($batchId)) {
            $query->where('batch_id', (int)$batchId);
        }

        $files = $query->limit($limit)->get();
        $this->info("Draining up to {$limit} file(s) " . ($batchId ? "from batch {$batchId}" : '') . "...");

        $processed = 0;
        foreach ($files as $file) {
            $this->line(" - Processing file {$file->id}: {$file->original_filename}");
            try {
                // Call the job handler directly to process synchronously
                (new ProcessCommissionFile($file))->handle();
                $processed++;
            } catch (\Throwable $e) {
                Log::error('commission:drain failed', ['file_id' => $file->id, 'error' => $e->getMessage()]);
                $this->error("   Error: {$e->getMessage()}");
            }
        }

        $this->info("Processed {$processed} file(s).");
        return Command::SUCCESS;
    }
}











































