<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\TotalCacheService;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;

class BackfillTotalCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'total-cache:backfill 
                            {--entity= : Specific entity type to backfill (shipment_line, customer_order_line, customer_order, shipment)}
                            {--id= : Specific entity ID to backfill}
                            {--chunk=100 : Number of records to process per chunk}
                            {--force : Force backfill even if cache exists}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Backfill total cache for existing records';

    protected TotalCacheService $service;

    public function __construct(TotalCacheService $service)
    {
        parent::__construct();
        $this->service = $service;
    }

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $entityType = $this->option('entity');
        $entityId = $this->option('id');
        $chunkSize = (int) $this->option('chunk');
        $force = $this->option('force');

        $this->info('Starting total cache backfill...');

        if ($entityType && $entityId) {
            $this->backfillSpecificEntity($entityType, (int) $entityId, $force);
        } elseif ($entityType) {
            $this->backfillEntityType($entityType, $chunkSize, $force);
        } else {
            $this->backfillAllEntities($chunkSize, $force);
        }

        $this->info('Total cache backfill completed!');
        
        // Show cache statistics
        $stats = $this->service->getCacheStats();
        $this->table(
            ['Metric', 'Count'],
            [
                ['Total Cache Entries', $stats['total']],
                ['Fresh Entries', $stats['fresh']],
                ['Stale Entries', $stats['stale']],
            ]
        );
    }

    protected function backfillSpecificEntity(string $entityType, int $entityId, bool $force): void
    {
        $this->info("Backfilling {$entityType} ID {$entityId}...");

        $entity = $this->getEntity($entityType, $entityId);
        if (!$entity) {
            $this->error("Entity {$entityType} with ID {$entityId} not found.");
            return;
        }

        $this->warmupEntity($entityType, $entity, $force);
        $this->info("✓ Completed {$entityType} ID {$entityId}");
    }

    protected function backfillEntityType(string $entityType, int $chunkSize, bool $force): void
    {
        $this->info("Backfilling all {$entityType} records...");

        $query = $this->getEntityQuery($entityType);
        $total = $query->count();
        
        if ($total === 0) {
            $this->warn("No {$entityType} records found.");
            return;
        }

        $this->info("Found {$total} {$entityType} records to process.");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $query->chunk($chunkSize, function ($entities) use ($entityType, $force, $bar) {
            foreach ($entities as $entity) {
                $this->warmupEntity($entityType, $entity, $force);
                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine();
    }

    protected function backfillAllEntities(int $chunkSize, bool $force): void
    {
        $entityTypes = ['shipment_line', 'customer_order_line', 'customer_order', 'shipment'];
        
        foreach ($entityTypes as $entityType) {
            $this->backfillEntityType($entityType, $chunkSize, $force);
        }
    }

    protected function getEntity(string $entityType, int $entityId)
    {
        $query = $this->getEntityQuery($entityType);
        return $query->find($entityId);
    }

    protected function getEntityQuery(string $entityType)
    {
        return match ($entityType) {
            'shipment_line' => ShipmentLine::with(['shipment_line_sizes', 'customer_order_lines.customer_order_line_quantities']),
            'customer_order_line' => CustomerOrderLines::with(['customer_order_line_quantities']),
            'customer_order' => CustomerOrders::with(['customer_order_lines.customer_order_line_quantities']),
            'shipment' => Shipment::with(['shipment_lines.customer_order_lines.customer_order_line_quantities']),
            default => throw new \InvalidArgumentException("Unknown entity type: {$entityType}")
        };
    }

    protected function warmupEntity(string $entityType, $entity, bool $force): void
    {
        try {
            // Check if cache already exists and we're not forcing
            if (!$force) {
                $existing = $this->service->get($entityType, $entity->id, 'prices');
                if ($existing) {
                    return; // Skip if cache exists
                }
            }

            match ($entityType) {
                'shipment_line' => $this->service->warmupShipmentLine($entity),
                'customer_order_line' => $this->service->warmupCustomerOrderLine($entity),
                'customer_order' => $this->service->warmupCustomerOrder($entity),
                'shipment' => $this->service->warmupShipment($entity),
            };
        } catch (\Exception $e) {
            $this->error("Failed to warmup {$entityType} ID {$entity->id}: " . $e->getMessage());
        }
    }
}