<?php

namespace App\Console\Commands;

use App\Models\CommodityCodes;
use App\Models\Styles;
use App\Services\VertexAiService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AssignCommodityCodes extends Command
{
    protected $signature = 'styles:assign-commodity-codes 
                            {--dry-run : Run without saving changes}
                            {--limit=5 : Number of styles to process (default 5 for testing)}
                            {--v : Verbose output - show detailed style info}';

    protected $description = 'Use AI to suggest commodity codes for styles without a commodity code assigned';

    protected VertexAiService $vertexAi;

    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        $limit = (int) $this->option('limit');

        $this->info($isDryRun ? '🔍 DRY RUN MODE - No changes will be saved' : '💾 LIVE MODE - Changes will be saved');
        $this->info("Processing up to {$limit} styles...\n");

        try {
            $this->vertexAi = new VertexAiService();
        } catch (\Exception $e) {
            $this->error('Failed to initialize Vertex AI: ' . $e->getMessage());
            return 1;
        }

        // Get all commodity codes for the prompt
        $commodityCodes = CommodityCodes::withoutGlobalScopes()
            ->select(['id', 'description', 'order', 'Gender', 'Type', 'Composition'])
            ->get()
            ->toArray();

        if (empty($commodityCodes)) {
            $this->error('No commodity codes found in the database.');
            return 1;
        }

        $this->info("Found " . count($commodityCodes) . " commodity codes in database.\n");

        // Find all styles that don't have a commodity code assigned
        $styles = Styles::withoutGlobalScopes()
            ->whereNull('commodity_codes_id')
            ->with([
                'designs',
                'customers',
                'departments',
                'seasons',
                'style_versions.colourways.colourway_yarns.yarn_colours.yarn.counts',
                'style_versions.colourways.colourway_yarns.yarn_colours.yarn.yarn_compositions',
            ])
            ->limit($limit)
            ->get();

        if ($styles->isEmpty()) {
            $this->info('No styles found that need commodity codes assigned.');
            return 0;
        }

        $total = $styles->count();
        $this->info("Found {$total} styles to process.");
        $this->info("Each style takes ~2-5 seconds to process via AI.\n");

        $successCount = 0;
        $failCount = 0;
        $current = 0;

        foreach ($styles as $style) {
            $current++;
            $startTime = microtime(true);
            
            $this->line("─────────────────────────────────────────");
            $designDesc = $style->designs?->description ?? 'N/A';
            $customerRef = $style->customer_ref ?? 'N/A';
            $this->info("[{$current}/{$total}] Style {$style->id}: {$designDesc} ({$customerRef})");
            
            $styleData = $this->prepareStyleData($style);
            
            if ($this->getOutput()->isVerbose()) {
                $this->table(
                    ['Field', 'Value'],
                    collect($styleData)->except('yarns')->map(fn($v, $k) => [$k, is_array($v) ? json_encode($v) : $v])->values()->toArray()
                );
                
                if (!empty($styleData['yarns'])) {
                    $this->line("\nYarns:");
                    foreach ($styleData['yarns'] as $yarn) {
                        $this->line("  - {$yarn}");
                    }
                }
            }

            try {
                $this->line("  → Calling Vertex AI...");
                $suggestedCodeId = $this->getSuggestedCommodityCode($styleData, $commodityCodes);
                
                if ($suggestedCodeId) {
                    $commodityCode = CommodityCodes::find($suggestedCodeId);
                    
                    if ($commodityCode) {
                        $this->info("  ✓ Matched: {$commodityCode->id}");
                        
                        if (!$isDryRun) {
                            $style->commodity_codes_id = $commodityCode->id;
                            $style->save();
                            $this->info("  ✓ Saved");
                        } else {
                            $this->warn("  (Dry run)");
                        }
                        
                        $successCount++;
                    } else {
                        $this->error("  ✗ ID {$suggestedCodeId} not found");
                        $failCount++;
                    }
                } else {
                    $this->warn("  ✗ No match");
                    $failCount++;
                }
            } catch (\Exception $e) {
                $this->error("  ✗ Error: " . $e->getMessage());
                Log::error("AssignCommodityCodes error for style {$style->id}", [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                $failCount++;
            }
            
            $elapsed = round(microtime(true) - $startTime, 1);
            $this->line("  ⏱ Completed in {$elapsed}s");
            $this->newLine();
        }

        $this->line("═════════════════════════════════════════");
        $this->info("Summary:");
        $this->line("  Successful: {$successCount}");
        $this->line("  Failed: {$failCount}");
        
        if ($isDryRun) {
            $this->newLine();
            $this->warn("This was a dry run. Run without --dry-run to save changes.");
        }

        return 0;
    }

    private function prepareStyleData(Styles $style): array
    {
        $yarns = [];
        $compositions = [];
        
        foreach ($style->style_versions as $version) {
            foreach ($version->colourways as $colourway) {
                foreach ($colourway->colourway_yarns as $colourwayYarn) {
                    $yarnColour = $colourwayYarn->yarn_colours;
                    if ($yarnColour && $yarnColour->yarn) {
                        $yarn = $yarnColour->yarn;
                        $count = $yarn->counts?->count ?? 'N/A';
                        $yarnDesc = "{$yarn->description} ({$count})";
                        
                        if (!in_array($yarnDesc, $yarns)) {
                            $yarns[] = $yarnDesc;
                        }
                        
                        // Gather composition info from yarn compositions if available
                        foreach ($yarn->yarn_compositions ?? [] as $comp) {
                            $compDesc = ($comp->material ?? '') . ' ' . ($comp->percentage ?? '') . '%';
                            if (!in_array($compDesc, $compositions) && trim($compDesc) !== '%') {
                                $compositions[] = trim($compDesc);
                            }
                        }
                    }
                }
                
                // Also check colourway composition if available
                if (!empty($colourway->composition)) {
                    $compositions[] = $colourway->composition;
                }
            }
        }

        return [
            'style_id' => $style->id,
            'design_id' => $style->designs_id,
            'design_description' => $style->designs?->description ?? 'N/A',
            'customer_ref' => $style->customer_ref ?? 'N/A',
            'customer' => $style->customers?->name ?? 'N/A',
            'category' => $style->category ?? 'N/A',
            'department' => $style->departments?->name ?? 'N/A',
            'season' => $style->seasons?->name ?? 'N/A',
            'yarns' => array_unique($yarns),
            'compositions' => array_unique($compositions),
            'notes' => $style->notes ?? '',
        ];
    }

    private function getSuggestedCommodityCode(array $styleData, array $commodityCodes): ?int
    {
        $prompt = $this->buildPrompt($styleData, $commodityCodes);
        
        $data = $this->vertexAi->generateText($prompt, 'gemini-2.5-flash-lite');
        
        if ($data && isset($data['commodity_code_id'])) {
            return (int) $data['commodity_code_id'];
        }
        
        return null;
    }

    private function buildPrompt(array $styleData, array $commodityCodes): string
    {
        $commodityCodesJson = json_encode($commodityCodes, JSON_PRETTY_PRINT);
        $styleDataJson = json_encode($styleData, JSON_PRETTY_PRINT);
        
        return <<<PROMPT
You are an expert in textile and garment classification for customs purposes. Your task is to select the most appropriate commodity code for a garment based on its characteristics.

## Garment Information:
{$styleDataJson}

## Available Commodity Codes:
{$commodityCodesJson}

## Instructions:
1. Analyze the garment information including the design description, yarns used, composition, category (mens/ladies/accessories/childrens), and any other relevant details.
2. Match the garment to the most appropriate commodity code from the list provided.
3. Consider:
   - The type of garment (jumper, cardigan, coat, dress, etc.)
   - The material composition (wool, cotton, synthetic, etc.)
   - Gender category (mens, ladies, childrens)
   - Any specific characteristics mentioned

## Response Format:
Return ONLY a valid JSON object with no additional text:
{
    "commodity_code_id": <the ID of the best matching commodity code>
}
PROMPT;
    }
}
