<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\TotalCache;
use App\Services\TotalCacheService;
use Illuminate\Support\Facades\Log;

class AddYarnValueEuroToTotalCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cache:add-yarn-value-euro {--dry-run : Show what would be done without actually doing it} {--recalculate : Recalculate all records instead of just adding missing fields}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Add yarn_value_euro field to existing total cache records';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        $recalculate = $this->option('recalculate');
        $totalCacheService = app(TotalCacheService::class);
        
        if ($dryRun) {
            $this->info('DRY RUN MODE - No changes will be made');
        }
        
        if ($recalculate) {
            $this->info('RECALCULATE MODE - Will recalculate all records that need yarn_value_euro');
            return $this->recalculateWithYarnValue($totalCacheService, $dryRun);
        } else {
            $this->info('UPDATE MODE - Will add yarn_value_euro field with calculated values');
            return $this->addYarnValueEuro($dryRun);
        }
    }
    
    /**
     * Add yarn_value_euro field to existing records by calculating from price_model data
     */
    private function addYarnValueEuro($dryRun)
    {
        $this->info('Adding yarn_value_euro field to existing total cache records...');
        
        $processedCount = 0;
        $updatedCount = 0;
        $errorCount = 0;
        
        TotalCache::chunk(100, function ($cacheRecords) use ($dryRun, &$processedCount, &$updatedCount, &$errorCount) {
            foreach ($cacheRecords as $cacheRecord) {
                $processedCount++;
                
                try {
                    $cachedData = $cacheRecord->cached_data;
                    $needsUpdate = false;
                    
                    // Check if yarn_value_euro is missing for line-level records
                    if (in_array($cacheRecord->entity_type, ['shipment_line', 'customer_order_line'])) {
                        if (!isset($cachedData['yarn_value_euro'])) {
                            $yarnValue = $this->calculateYarnValueForRecord($cacheRecord);
                            $cachedData['yarn_value_euro'] = $yarnValue;
                            $needsUpdate = true;
                        }
                    }
                    
                    // Check if yarn_value_euro_total is missing for shipment records
                    if ($cacheRecord->entity_type === 'shipment') {
                        if (!isset($cachedData['yarn_value_euro_total'])) {
                            $yarnValueTotal = $this->calculateYarnValueTotalForShipment($cacheRecord);
                            $cachedData['yarn_value_euro_total'] = $yarnValueTotal;
                            $needsUpdate = true;
                        }
                    }
                    
                    if ($needsUpdate && !$dryRun) {
                        $cacheRecord->update(['cached_data' => $cachedData]);
                        $updatedCount++;
                        
                        if ($updatedCount % 50 == 0) {
                            $this->line("Updated {$updatedCount} records...");
                        }
                    } elseif ($needsUpdate && $dryRun) {
                        $updatedCount++;
                        $this->line("Would update record {$cacheRecord->id} ({$cacheRecord->entity_type})");
                    }
                    
                } catch (\Exception $e) {
                    $errorCount++;
                    $this->error("Failed to update record {$cacheRecord->id}: " . $e->getMessage());
                }
                
                if ($processedCount % 1000 == 0) {
                    $this->info("Processed {$processedCount} records, updated {$updatedCount}, errors {$errorCount}");
                }
            }
        });
        
        $this->info("Completed! Processed: {$processedCount}, Updated: {$updatedCount}, Errors: {$errorCount}");
        
        // Show some sample updated records
        if (!$dryRun && $updatedCount > 0) {
            $this->info('Sample updated records:');
            $samples = TotalCache::whereIn('entity_type', ['customer_order_line', 'shipment_line', 'shipment'])
                ->whereNotNull('cached_data->yarn_value_euro')
                ->orWhereNotNull('cached_data->yarn_value_euro_total')
                ->limit(3)
                ->get(['id', 'entity_type', 'cached_data']);
                
            foreach ($samples as $sample) {
                $data = $sample->cached_data;
                $yarnValue = $data['yarn_value_euro'] ?? $data['yarn_value_euro_total'] ?? 'missing';
                $this->line("Record {$sample->id} ({$sample->entity_type}): yarn_value_euro = {$yarnValue}");
            }
        }
        
        return $errorCount > 0 ? Command::FAILURE : Command::SUCCESS;
    }
    
    /**
     * Recalculate records that need yarn_value_euro
     */
    private function recalculateWithYarnValue(TotalCacheService $totalCacheService, $dryRun)
    {
        $this->info('Recalculating records to include yarn_value_euro...');
        
        $processedCount = 0;
        $recalculatedCount = 0;
        $errorCount = 0;
        
        TotalCache::chunk(100, function ($cacheRecords) use ($totalCacheService, $dryRun, &$processedCount, &$recalculatedCount, &$errorCount) {
            foreach ($cacheRecords as $cacheRecord) {
                $processedCount++;
                
                try {
                    $cachedData = $cacheRecord->cached_data;
                    $needsRecalculation = false;
                    
                    // Check if yarn_value_euro is missing for line-level records
                    if (in_array($cacheRecord->entity_type, ['shipment_line', 'customer_order_line'])) {
                        if (!isset($cachedData['yarn_value_euro'])) {
                            $needsRecalculation = true;
                        }
                    }
                    
                    // Check if yarn_value_euro_total is missing for shipment records
                    if ($cacheRecord->entity_type === 'shipment') {
                        if (!isset($cachedData['yarn_value_euro_total'])) {
                            $needsRecalculation = true;
                        }
                    }
                    
                    if ($needsRecalculation && !$dryRun) {
                        $this->recalculateRecord($cacheRecord, $totalCacheService);
                        $recalculatedCount++;
                        
                        if ($recalculatedCount % 50 == 0) {
                            $this->line("Recalculated {$recalculatedCount} records...");
                        }
                    } elseif ($needsRecalculation && $dryRun) {
                        $recalculatedCount++;
                        $this->line("Would recalculate record {$cacheRecord->id} ({$cacheRecord->entity_type})");
                    }
                    
                } catch (\Exception $e) {
                    $errorCount++;
                    $this->error("Failed to recalculate record {$cacheRecord->id}: " . $e->getMessage());
                }
                
                if ($processedCount % 1000 == 0) {
                    $this->info("Processed {$processedCount} records, recalculated {$recalculatedCount}, errors {$errorCount}");
                }
            }
        });
        
        $this->info("Completed! Processed: {$processedCount}, Recalculated: {$recalculatedCount}, Errors: {$errorCount}");
        
        return $errorCount > 0 ? Command::FAILURE : Command::SUCCESS;
    }
    
    /**
     * Calculate yarn_value_euro for a specific cache record
     */
    private function calculateYarnValueForRecord(TotalCache $cacheRecord)
    {
        $yarnValue = 0;
        
        try {
            if ($cacheRecord->entity_type === 'shipment_line') {
                $shipmentLine = \App\Models\ShipmentLine::find($cacheRecord->entity_id);
                if ($shipmentLine) {
                    foreach ($shipmentLine->shipment_line_sizes as $qty) {
                        $price = $shipmentLine->customer_order_lines->customer_order_line_quantities
                            ->where('sizes_id', $qty->sizes_id)
                            ->first()?->price_model;
                        
                        if (!empty($price) && isset($price['yarn_cost_kg'])) {
                            $pieces = empty($qty->shipped_qty) ? $qty->qty : $qty->shipped_qty;
                            $yarnValue += ($price['yarn_cost_kg'] * $pieces);
                        }
                    }
                }
            } elseif ($cacheRecord->entity_type === 'customer_order_line') {
                $orderLine = \App\Models\CustomerOrderLines::find($cacheRecord->entity_id);
                if ($orderLine) {
                    foreach ($orderLine->customer_order_line_quantities as $qty) {
                        $price = $qty->price_model;
                        if (!empty($price) && isset($price['yarn_cost_kg'])) {
                            $pieces = $qty->qty;
                            $yarnValue += ($price['yarn_cost_kg'] * $pieces);
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            Log::error("Failed to calculate yarn value for record {$cacheRecord->id}: " . $e->getMessage());
        }
        
        return $yarnValue;
    }
    
    /**
     * Calculate yarn_value_euro_total for a shipment
     */
    private function calculateYarnValueTotalForShipment(TotalCache $cacheRecord)
    {
        $yarnValueTotal = 0;
        
        try {
            $shipment = \App\Models\Shipment::find($cacheRecord->entity_id);
            if ($shipment) {
                foreach ($shipment->shipment_lines as $sl) {
                    // Get price model data for this line
                    $priceModel = $sl->customer_order_lines->price_model;
                    
                    // Check if this shipment line has any shipped quantities at all
                    $hasShippedQuantities = $sl->shipment_line_sizes->whereNotNull('shipped_qty')->count() > 0;
                    
                    foreach ($sl->customer_order_lines->customer_order_line_quantities as $colq) {
                        // If shipment line has any shipped quantities, use shipped for all sizes (even 0/null)
                        // Otherwise, use ordered quantities
                        if ($hasShippedQuantities) {
                            $shippedQty = $sl->shipment_line_sizes->where('sizes_id', $colq->sizes_id)->first()?->shipped_qty;
                            $qty = $shippedQty ?? 0; // Use shipped qty even if null (treat as 0)
                        } else {
                            $qty = $colq->qty; // Use ordered qty
                        }
                        
                        if ($priceModel && isset($priceModel['yarn_cost_kg'])) {
                            $yarnValueTotal += $priceModel['yarn_cost_kg'] * $qty;
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            Log::error("Failed to calculate yarn value total for shipment {$cacheRecord->id}: " . $e->getMessage());
        }
        
        return $yarnValueTotal;
    }
    
    /**
     * Recalculate a specific cache record
     */
    private function recalculateRecord(TotalCache $cacheRecord, TotalCacheService $totalCacheService)
    {
        $entityType = $cacheRecord->entity_type;
        $entityId = $cacheRecord->entity_id;
        $cacheKey = $cacheRecord->cache_key;
        
        // Invalidate existing cache to force recalculation
        $totalCacheService->invalidateEntity($entityType, $entityId, $cacheKey);
        
        // Warm up with new calculation that includes yarn_value_euro
        switch ($entityType) {
            case 'shipment_line':
                $entity = \App\Models\ShipmentLine::find($entityId);
                if ($entity) {
                    $totalCacheService->warmupShipmentLine($entity);
                }
                break;
                
            case 'customer_order_line':
                $entity = \App\Models\CustomerOrderLines::find($entityId);
                if ($entity) {
                    $totalCacheService->warmupCustomerOrderLine($entity);
                }
                break;
                
            case 'customer_order':
                $entity = \App\Models\CustomerOrders::find($entityId);
                if ($entity) {
                    $totalCacheService->warmupCustomerOrder($entity);
                }
                break;
                
            case 'shipment':
                $entity = \App\Models\Shipment::find($entityId);
                if ($entity) {
                    $totalCacheService->warmupShipment($entity);
                }
                break;
                
            default:
                throw new \Exception("Unknown entity type: {$entityType}");
        }
    }
}


