<?php

namespace App\Actions;

use Illuminate\Support\Collection;
use App\Models\ShipmentLine;
use App\Models\InvoiceLayout;
use App\ValueObjects\ZohoInvoice;

/**
 * Build ZohoInvoice value objects from completed shipment-lines.
 *
 * Usage:
 *   $invoices = GenerateInvoices::run(customerId: 42);   // returns Collection
 */
class GenerateInvoices
{
    /** @return Collection<ZohoInvoice> */
    public static function run(?int $customerId = null): Collection
    {
        // ---------- 1.  Load candidate shipment-lines -----------------------
        $query = ShipmentLine::with([
            'customer_order_lines.customer_orders.customers.customer_invoice_layouts.invoice_layouts',
            'customer_order_lines.colourways.style_versions.styles.designs',
            'customer_order_lines.customer_order_line_quantities',
            'shipment_line_sizes.sizes',
            'customer_order_lines.customer_orders.departments',
            'customer_order_lines.customer_orders.customer_addresses',
        ])
        ->whereRelation('customer_order_lines.customer_orders', 'order_type', 'wholesale')
        ->whereRelation('customer_order_lines.customer_orders', 'cancelled', 0)
        ->where('complete', 1)
        ->where(fn ($q) => $q->whereNull('shipment_lines.rt_invoice')->orWhere('shipment_lines.rt_invoice', ''));

        if ($customerId) {
            $query->whereRelation(
                'customer_order_lines.customer_orders.customers',
                'id',
                $customerId
            );
        }

        /** @var Collection<ShipmentLine> $lines */
        $lines = $query->get()->groupBy('customer_order_lines.customer_orders.customers_id');

        // ---------- 2.  Transform into ZohoInvoice DTOs ----------------------
        return $lines->flatMap(function (Collection $customerGroup) {
            $customer      = $customerGroup->first()->customer_order_lines->customer_orders->customers;
            $firstShipment = $customerGroup->first();
            $incoterm      = $firstShipment->customer_order_lines->customer_orders->incoterms;

            $layout = $customer->customer_invoice_layouts
                        ->where('incoterm', $incoterm)
                        ->first()
                        ->invoice_layouts
                        ?? InvoiceLayout::where('default', 1)->first();

            // Decide how to group shipments within this customer
            if ($layout->separate_invoice_per_collection_date) {
                // When separate_invoice_per_collection_date is enabled, group by collection date only
                $grouped = $customerGroup->groupBy(function ($shipment) {
                    return $shipment->collection_date ? $shipment->collection_date->format('Y-m-d') : 'no-collection-date';
                });
            } else {
                // Use the original grouping logic
                $grouped = match ($layout->invoice_by) {
                    'order'   => $customerGroup->groupBy('customer_order_lines.customer_orders_id'),
                    'style'   => $customerGroup->groupBy('customer_order_lines.colourways.style_versions.styles_id'),
                    'factory' => $customerGroup->groupBy('customer_order_lines.colourways.style_versions.factory_id'),
                    'drop'    => $customerGroup->groupBy('id'),
                    default   => $customerGroup->groupBy('collection_date'),
                };
            }

            // Build one ZohoInvoice per group
            return $grouped->map(function (Collection $shipments) use ($layout) {
                // Collect all unique POs from the shipments group
                $allPOs = $shipments
                    ->pluck('customer_order_lines.customer_orders.customer_po')
                    ->filter()
                    ->unique()
                    ->values()
                    ->implode(', ');
                    
                $invoice = new ZohoInvoice(
                    customerName        : self::determineCustomerName($shipments),
                    invoiceDate         : self::determineInvoiceDate($shipments, $layout),
                    invoicePO           : $allPOs,
                    invoiceLayout       : $layout->invoice_layout,
                    invoiceComments     : self::format($layout->invoice_comments, $shipments->first()),
                    invoiceShippingAddr : $shipments->first()->customer_order_lines->customer_orders->customer_addresses->name,
                    zohoCustomerTag     : self::customerTag($shipments),
                    zohoAccount         : $layout->zoho_account,
                    zohoDepartment      : $shipments->first()->customer_order_lines->customer_orders->departments->zoho_name,
                    zohoFactored        : $shipments->first()->customer_order_lines->customer_orders->customers->zoho_factored,
                );

                foreach ($shipments as $drop) {
                    // Check if drop has valid quantity before adding to invoice
                    $lineItem = self::makeLineItem($drop, $layout);
                    $quantity = $lineItem['qty'] ?? 1;
                    
                    if ($quantity > 0) {
                        $invoice->addLine($lineItem);
                        $invoice->drops[] = $drop->id;
                    }
                }

                return $invoice;
            });
        })->values();
    }

    /* ---------- tiny private helpers (pure functions) ------------------- */

    private static function determineCustomerName(Collection $shipments): string
    {
        $customer = $shipments->first()->customer_order_lines->customer_orders->customers;
        return $customer->zoho_customer ?? $customer->name;
    }

    private static function determineInvoiceDate(Collection $shipments, $layout = null): string
    {
        $first = $shipments->first();
        
        // If separate_invoice_per_collection_date is enabled, use collection date
        if ($layout && $layout->separate_invoice_per_collection_date) {
            return optional($first->collection_date)->format('Y-m-d');
        }
        
        // Otherwise use the original logic
        return optional($first->collection_date ?? $first->exfty)->format('Y-m-d');
    }

    private static function customerTag(Collection $shipments): string
    {
        $customer = $shipments->first()->customer_order_lines->customer_orders->customers;
        return $customer->zoho_customer_tag
            ?? $customer->zoho_customer
            ?? $customer->name;
    }

    private static function format(string $template, $drop): string
    {
        // Very small replacement helper – could be extracted if reused elsewhere.
        return preg_replace_callback(
            '/\{([^}]+)\}/',
            fn ($m) => data_get($drop, $m[1], ''),
            $template
        );
    }

    /** Builds one line-item array for a given ShipmentLine */
    private static function makeLineItem($drop, $layout): array
    {
        // Condensed: plug in your own field-mapping / VAT logic here.
        return [
            'item_title'     => self::format($layout->line_title, $drop),
            'lineDescription'=> self::format($layout->line_description, $drop),
            'price'          => self::format($layout->line_price, $drop),
            'qty'            => self::format($layout->line_qty, $drop),
            'vat'            => true,
        ];
    }
}
