<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Update SP Page Builder Page Action
 *
 * @since  1.0.0
 */
class UpdatePageBuilderPageAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Update result
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for UpdatePageBuilderPageAction');
        }

        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }
        $pageId = $parameters['page_id'];

        // Check if SP Page Builder is installed
        if (!file_exists(JPATH_ROOT . '/components/com_sppagebuilder')) {
            throw new \RuntimeException('SP Page Builder is not installed');
        }

        // Get current page
        $query = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__sppagebuilder_pages'))
            ->where($db->quoteName('id') . ' = ' . (int) $pageId);

        $db->setQuery($query);
        $page = $db->loadObject();

        if (!$page) {
            throw new \RuntimeException("SP Page Builder page with ID {$pageId} not found");
        }

        // Prepare update query
        $updateFields = [];

        if (isset($parameters['title'])) {
            $updateFields[] = $db->quoteName('title') . ' = ' . $db->quote($parameters['title']);
        }

        if (isset($parameters['content'])) {
            $content = is_array($parameters['content']) 
                ? json_encode($parameters['content']) 
                : $parameters['content'];
            $updateFields[] = $db->quoteName('text') . ' = ' . $db->quote($content);
        }

        if (isset($parameters['published'])) {
            $updateFields[] = $db->quoteName('published') . ' = ' . (int) $parameters['published'];
        }

        if (empty($updateFields)) {
            throw new \InvalidArgumentException('No valid fields to update');
        }

        // Add modified date
        $updateFields[] = $db->quoteName('modified') . ' = ' . $db->quote(Factory::getDate()->toSql());

        // Execute update
        $query = $db->getQuery(true)
            ->update($db->quoteName('#__sppagebuilder_pages'))
            ->set($updateFields)
            ->where($db->quoteName('id') . ' = ' . (int) $pageId);

        $db->setQuery($query);
        $db->execute();

        return [
            'success' => true,
            'page_id' => $pageId,
            'updated_fields' => array_keys(array_intersect_key($parameters, array_flip(['title', 'content', 'published']))),
            'message' => 'Page Builder page updated successfully'
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Update a SP Page Builder page. Required: page_id. Optional: title, content (JSON array of sections), published.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['page_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['page_id']) && is_numeric($parameters['page_id']);
    }
}

