<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;

defined('_JEXEC') or die;

/**
 * OpenAI API Provider
 *
 * @since  1.0.0
 */
class OpenAIProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * API endpoint
     *
     * @var    string
     * @since  1.0.0
     */
    private string $endpoint = 'https://api.openai.com/v1/chat/completions';

    /**
     * Constructor
     *
     * @param   string  $apiKey  OpenAI API key
     * @param   string  $model   Model name
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'gpt-4o')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Send a chat message and get response
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        $http = HttpFactory::getHttp();

        $payload = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => 0.7,
            'max_tokens' => 4000
        ];

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        try {
            $response = $http->post(
                $this->endpoint,
                json_encode($payload),
                $headers,
                30
            );

            if ($response->code !== 200) {
                throw new \RuntimeException(
                    "OpenAI API error: {$response->code} - {$response->body}"
                );
            }

            $data = json_decode($response->body, true);

            if (!isset($data['choices'][0]['message']['content'])) {
                throw new \RuntimeException('Invalid response from OpenAI API');
            }

            return $data['choices'][0]['message']['content'];

        } catch (\Exception $e) {
            throw new \RuntimeException(
                'Failed to communicate with OpenAI: ' . $e->getMessage()
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'OpenAI (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

