<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * OpenAI API Provider
 *
 * @since  1.0.0
 */
class OpenAIProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * Constructor
     *
     * @param   string  $apiKey  OpenAI API key
     * @param   string  $model   Model name
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'gpt-5')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Send a chat message and get response using Responses API
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        // Validate API key is set
        if (empty($this->apiKey)) {
            Log::add(
                "OpenAI API key is empty!",
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException('OpenAI API key is not configured. Please add your API key in plugin settings.');
        }
        
        if (strlen($this->apiKey) < 50) {
            Log::add(
                "OpenAI API key is too short: " . strlen($this->apiKey) . " characters",
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException('OpenAI API key appears to be truncated (only ' . strlen($this->apiKey) . ' characters). Please re-enter your full API key.');
        }

        $http = HttpFactory::getHttp();

        // For simple single message, use string input
        // For multi-turn, build a simple string from messages
        $input = '';
        $systemPrompt = '';
        
        foreach ($messages as $msg) {
            if ($msg['role'] === 'system') {
                $systemPrompt = $msg['content'];
            } elseif ($msg['role'] === 'user') {
                $input = $msg['content'];
            } elseif ($msg['role'] === 'assistant') {
                // For conversation history, concatenate
                $input .= "\nAssistant: " . $msg['content'] . "\n";
            }
        }
        
        // Prepend system prompt if exists
        if (!empty($systemPrompt)) {
            $input = $systemPrompt . "\n\n" . $input;
        }

        // Build payload for Responses API - use simple string input
        $payload = [
            'model' => $this->model,
            'input' => $input,
            'reasoning' => ['effort' => 'medium'],
            'text' => ['verbosity' => 'medium'],
            'max_output_tokens' => 4000
        ];

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        // Log the request for debugging
        Log::add(
            "OpenAI Request: Model={$this->model}, Endpoint=https://api.openai.com/v1/responses, API Key Length=" . strlen($this->apiKey),
            Log::INFO,
            'aiassistant'
        );
        
        // Log payload structure (without sensitive data)
        Log::add(
            "OpenAI Payload: " . json_encode(['model' => $this->model, 'input_length' => strlen($input)]),
            Log::INFO,
            'aiassistant'
        );

        try {
            Log::add('[DEBUG] About to send request to OpenAI API (timeout: 120s)', Log::DEBUG, 'aiassistant');
            
            $response = $http->post(
                'https://api.openai.com/v1/responses',
                json_encode($payload),
                $headers,
                120  // Increased from 60 to 120 seconds for longer responses
            );

            Log::add('[DEBUG] OpenAI API request completed', Log::DEBUG, 'aiassistant');
            
            // Log response code
            Log::add(
                "OpenAI Response Code: {$response->code}",
                Log::INFO,
                'aiassistant'
            );

            if ($response->code !== 200) {
                // Log full error for debugging
                Log::add(
                    "OpenAI Responses API error: {$response->code} - {$response->body}",
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "OpenAI API error (HTTP {$response->code}): " . substr($response->body, 0, 200)
                );
            }

            // Log the raw response for debugging
            Log::add('[DEBUG] OpenAI raw response: ' . substr($response->body, 0, 1000), Log::DEBUG, 'aiassistant');
            
            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::add(
                    "[ERROR] Invalid JSON from OpenAI: " . $response->body,
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('Invalid JSON response from OpenAI');
            }
            
            // Log the decoded structure for debugging
            Log::add('[DEBUG] OpenAI decoded response structure: ' . json_encode(array_keys($data)), Log::DEBUG, 'aiassistant');

            // Extract text from the Responses API output structure
            // Structure: output[] -> find type="message" -> content[] -> find type="output_text" -> text
            if (!isset($data['output']) || !is_array($data['output'])) {
                Log::add(
                    "[ERROR] No output in OpenAI response. Full response: " . json_encode($data),
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('No output in OpenAI response. Check logs for full response structure.');
            }

            Log::add('[DEBUG] OpenAI output array has ' . count($data['output']) . ' items', Log::DEBUG, 'aiassistant');

            // Find the message output
            foreach ($data['output'] as $index => $output) {
                Log::add('[DEBUG] Output[' . $index . '] type: ' . ($output['type'] ?? 'none'), Log::DEBUG, 'aiassistant');
                
                if (isset($output['type']) && $output['type'] === 'message') {
                    if (isset($output['content']) && is_array($output['content'])) {
                        Log::add('[DEBUG] Message has ' . count($output['content']) . ' content items', Log::DEBUG, 'aiassistant');
                        
                        foreach ($output['content'] as $contentIndex => $content) {
                            Log::add('[DEBUG] Content[' . $contentIndex . '] type: ' . ($content['type'] ?? 'none'), Log::DEBUG, 'aiassistant');
                            
                            if (isset($content['type']) && $content['type'] === 'output_text') {
                                $text = $content['text'] ?? '';
                                Log::add('[DEBUG] Found output_text, length: ' . strlen($text), Log::DEBUG, 'aiassistant');
                                return $text;
                            }
                        }
                    }
                }
            }

            // If we get here, couldn't find the text
            Log::add(
                "[ERROR] Could not extract text from OpenAI response. Full structure: " . json_encode($data, JSON_PRETTY_PRINT),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException('Could not extract text from OpenAI response. The response structure may have changed or be different than expected.');

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::add(
                "[ERROR] OpenAI Responses API error: " . $e->getMessage() . " | Code: " . $e->getCode() . " | File: " . $e->getFile() . " | Line: " . $e->getLine(),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with OpenAI Responses API. ' . $e->getMessage() . ' (Check your API key and model name in plugin settings)'
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'OpenAI (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

