<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * OpenAI API Provider
 *
 * @since  1.0.0
 */
class OpenAIProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * Constructor
     *
     * @param   string  $apiKey  OpenAI API key
     * @param   string  $model   Model name (default: gpt-4o)
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'gpt-4o')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Send a chat message and get response using Chat Completions API
     *
     * @param   array  $messages  Conversation history (array of messages with role and content)
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        // Validate API key is set
        if (empty($this->apiKey)) {
            Log::add(
                "OpenAI API key is empty!",
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException('OpenAI API key is not configured. Please add your API key in plugin settings.');
        }
        
        if (strlen($this->apiKey) < 50) {
            Log::add(
                "OpenAI API key is too short: " . strlen($this->apiKey) . " characters",
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException('OpenAI API key appears to be truncated (only ' . strlen($this->apiKey) . ' characters). Please re-enter your full API key.');
        }

        $http = HttpFactory::getHttp();

        // Build payload for Chat Completions API (standard for all models)
        // Note: Some models (o1, gpt-5-*) use different parameter names
        $isNewFormat = (
            strpos($this->model, 'o1') === 0 || 
            strpos($this->model, 'gpt-5') === 0
        );
        
        $payload = [
            'model' => $this->model,
            'messages' => $messages
        ];
        
        // Use the appropriate token limit parameter based on model
        if ($isNewFormat) {
            // Newer models (o1, gpt-5) use max_completion_tokens
            $payload['max_completion_tokens'] = 4000;
            // Note: o1 models don't support temperature, but gpt-5 might
            if (strpos($this->model, 'gpt-5') === 0) {
                $payload['temperature'] = 0.7;
            }
        } else {
            // Standard models (gpt-4, gpt-4o, gpt-3.5) use max_tokens
            $payload['max_tokens'] = 4000;
            $payload['temperature'] = 0.7;
        }

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        // Log the request for debugging
        Log::add(
            "OpenAI Request: Model={$this->model}, Endpoint=https://api.openai.com/v1/chat/completions, API Key Length=" . strlen($this->apiKey),
            Log::INFO,
            'aiassistant'
        );
        
        // Log payload structure (without sensitive data)
        Log::add(
            "OpenAI Payload: " . json_encode(['model' => $this->model, 'message_count' => count($messages)]),
            Log::INFO,
            'aiassistant'
        );

        try {
            Log::add('[DEBUG] About to send request to OpenAI API (timeout: 120s)', Log::DEBUG, 'aiassistant');
            
            $response = $http->post(
                'https://api.openai.com/v1/chat/completions',
                json_encode($payload),
                $headers,
                120  // Increased from 60 to 120 seconds for longer responses
            );

            Log::add('[DEBUG] OpenAI API request completed', Log::DEBUG, 'aiassistant');
            
            // Log response code
            Log::add(
                "OpenAI Response Code: {$response->code}",
                Log::INFO,
                'aiassistant'
            );

            if ($response->code !== 200) {
                // Log full error for debugging
                Log::add(
                    "OpenAI Chat API error: {$response->code} - {$response->body}",
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "OpenAI API error (HTTP {$response->code}): " . substr($response->body, 0, 200)
                );
            }

            // Log the raw response for debugging
            Log::add('[DEBUG] OpenAI raw response: ' . substr($response->body, 0, 1000), Log::DEBUG, 'aiassistant');
            
            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::add(
                    "[ERROR] Invalid JSON from OpenAI: " . $response->body,
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('Invalid JSON response from OpenAI');
            }
            
            // Log the decoded structure for debugging
            Log::add('[DEBUG] OpenAI decoded response structure: ' . json_encode(array_keys($data)), Log::DEBUG, 'aiassistant');

            // Extract message from Chat Completions API response
            // Standard format: choices[0].message.content
            if (!isset($data['choices']) || !is_array($data['choices']) || empty($data['choices'])) {
                Log::add(
                    "[ERROR] No choices in OpenAI response. Full response: " . json_encode($data),
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('No choices in OpenAI response. Check logs for full response structure.');
            }

            $firstChoice = $data['choices'][0] ?? null;
            if (!$firstChoice || !isset($firstChoice['message']['content'])) {
                Log::add(
                    "[ERROR] Invalid choice structure in OpenAI response. Full response: " . json_encode($data),
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('Invalid message structure in OpenAI response.');
            }

            $text = $firstChoice['message']['content'];
            Log::add('[DEBUG] Successfully extracted message content, length: ' . strlen($text), Log::DEBUG, 'aiassistant');
            
            return $text;

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::add(
                "[ERROR] OpenAI Responses API error: " . $e->getMessage() . " | Code: " . $e->getCode() . " | File: " . $e->getFile() . " | Line: " . $e->getLine(),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with OpenAI Responses API. ' . $e->getMessage() . ' (Check your API key and model name in plugin settings)'
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'OpenAI (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

