<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * OpenAI API Provider
 *
 * @since  1.0.0
 */
class OpenAIProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * Constructor
     *
     * @param   string  $apiKey  OpenAI API key
     * @param   string  $model   Model name
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'gpt-5')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Send a chat message and get response using Responses API
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        // Validate API key is set
        if (empty($this->apiKey)) {
            throw new \RuntimeException('OpenAI API key is not configured');
        }

        $http = HttpFactory::getHttp();

        // For simple single message, use string input
        // For multi-turn, build a simple string from messages
        $input = '';
        $systemPrompt = '';
        
        foreach ($messages as $msg) {
            if ($msg['role'] === 'system') {
                $systemPrompt = $msg['content'];
            } elseif ($msg['role'] === 'user') {
                $input = $msg['content'];
            } elseif ($msg['role'] === 'assistant') {
                // For conversation history, concatenate
                $input .= "\nAssistant: " . $msg['content'] . "\n";
            }
        }
        
        // Prepend system prompt if exists
        if (!empty($systemPrompt)) {
            $input = $systemPrompt . "\n\n" . $input;
        }

        // Build payload for Responses API - use simple string input
        $payload = [
            'model' => $this->model,
            'input' => $input,
            'reasoning' => ['effort' => 'medium'],
            'text' => ['verbosity' => 'medium'],
            'max_output_tokens' => 4000
        ];

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        // Log the request for debugging
        Log::add(
            "OpenAI Request: Model={$this->model}, Endpoint=https://api.openai.com/v1/responses",
            Log::INFO,
            'aiassistant'
        );

        try {
            $response = $http->post(
                'https://api.openai.com/v1/responses',
                json_encode($payload),
                $headers,
                60
            );

            // Log response code
            Log::add(
                "OpenAI Response Code: {$response->code}",
                Log::INFO,
                'aiassistant'
            );

            if ($response->code !== 200) {
                // Log full error for debugging
                Log::add(
                    "OpenAI Responses API error: {$response->code} - {$response->body}",
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "OpenAI API error (HTTP {$response->code}): " . substr($response->body, 0, 200)
                );
            }

            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE || !isset($data['output_text'])) {
                Log::add(
                    "Invalid OpenAI response: " . $response->body,
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException('Invalid response from OpenAI Responses API');
            }

            return $data['output_text'];

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::add(
                'OpenAI Responses API error: ' . $e->getMessage(),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with OpenAI Responses API. Check error logs for details.'
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'OpenAI (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

