<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * OpenAI API Provider
 *
 * @since  1.0.0
 */
class OpenAIProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * Constructor
     *
     * @param   string  $apiKey  OpenAI API key
     * @param   string  $model   Model name
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'gpt-5')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Check if model uses the new Responses API (GPT-5 family)
     *
     * @return  boolean
     * @since   1.0.0
     */
    private function isResponsesAPI(): bool
    {
        return str_starts_with($this->model, 'gpt-5');
    }

    /**
     * Send a chat message and get response
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        // Validate API key is set
        if (empty($this->apiKey)) {
            throw new \RuntimeException('OpenAI API key is not configured');
        }

        // Use different API based on model
        if ($this->isResponsesAPI()) {
            return $this->chatResponsesAPI($messages);
        } else {
            return $this->chatCompletionsAPI($messages);
        }
    }

    /**
     * Chat using the new Responses API (for GPT-5 models)
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    private function chatResponsesAPI(array $messages): string
    {
        $http = HttpFactory::getHttp();

        // Convert messages to single input for Responses API
        $input = '';
        foreach ($messages as $msg) {
            if ($msg['role'] === 'system') {
                $input .= "System: " . $msg['content'] . "\n\n";
            } elseif ($msg['role'] === 'user') {
                $input .= $msg['content'];
            }
        }

        $payload = [
            'model' => $this->model,
            'input' => $input,
            'reasoning' => ['effort' => 'medium'],
            'text' => ['verbosity' => 'medium'],
            'max_output_tokens' => 4000
        ];

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        try {
            $response = $http->post(
                'https://api.openai.com/v1/responses',
                json_encode($payload),
                $headers,
                60
            );

            if ($response->code !== 200) {
                Log::add(
                    "OpenAI Responses API error: {$response->code} - {$response->body}",
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "OpenAI API error (HTTP {$response->code}). Check error logs for details."
                );
            }

            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE || !isset($data['output_text'])) {
                throw new \RuntimeException('Invalid response from OpenAI Responses API');
            }

            return $data['output_text'];

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::add(
                'OpenAI Responses API error: ' . $e->getMessage(),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with OpenAI Responses API. Check error logs for details.'
            );
        }
    }

    /**
     * Chat using the legacy Chat Completions API (for GPT-4, GPT-3.5, etc.)
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    private function chatCompletionsAPI(array $messages): string
    {
        $http = HttpFactory::getHttp();

        $payload = [
            'model' => $this->model,
            'messages' => $messages,
            'temperature' => 0.7,
            'max_tokens' => 4000
        ];

        $headers = [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $this->apiKey
        ];

        try {
            $response = $http->post(
                'https://api.openai.com/v1/chat/completions',
                json_encode($payload),
                $headers,
                30
            );

            if ($response->code !== 200) {
                Log::add(
                    "OpenAI Chat Completions API error: {$response->code} - {$response->body}",
                    Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "OpenAI API error (HTTP {$response->code}). Check error logs for details."
                );
            }

            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE || !isset($data['choices'][0]['message']['content'])) {
                throw new \RuntimeException('Invalid response from OpenAI Chat Completions API');
            }

            return $data['choices'][0]['message']['content'];

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::add(
                'OpenAI Chat Completions API error: ' . $e->getMessage(),
                Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with OpenAI Chat Completions API. Check error logs for details.'
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'OpenAI (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

