const path = require('path');
const fs = require('fs');
const sharp = require('sharp');
const db = require('./db');

// Generate thumbnail for an image
async function generateThumbnail(albumId, filename) {
  const sourcePath = path.join(__dirname, '..', 'uploads', String(albumId), filename);
  const thumbsDir = path.join(__dirname, '..', 'uploads', String(albumId), 'thumbs');
  const thumbPath = path.join(thumbsDir, filename);
  
  // Skip if thumbnail already exists
  if (fs.existsSync(thumbPath)) {
    console.log(`  ✓ Thumbnail already exists: ${filename}`);
    return true;
  }
  
  // Skip if source doesn't exist
  if (!fs.existsSync(sourcePath)) {
    console.log(`  ✗ Source not found: ${filename}`);
    return false;
  }
  
  // Create thumbs directory if it doesn't exist
  fs.mkdirSync(thumbsDir, { recursive: true });
  
  try {
    // Generate thumbnail (max 600px width for high-res displays, maintain aspect ratio)
    await sharp(sourcePath)
      .resize(600, null, {
        withoutEnlargement: true,
        fit: 'inside'
      })
      .jpeg({ quality: 92, progressive: true })
      .toFile(thumbPath);
    
    console.log(`  ✓ Generated thumbnail: ${filename}`);
    return true;
  } catch (error) {
    console.error(`  ✗ Error generating thumbnail for ${filename}:`, error.message);
    return false;
  }
}

async function generateAllThumbnails() {
  console.log('Starting thumbnail generation for all existing photos...\n');
  
  const albums = db.listAlbums();
  
  if (albums.length === 0) {
    console.log('No albums found.');
    return;
  }
  
  let totalProcessed = 0;
  let totalSuccess = 0;
  let totalSkipped = 0;
  let totalFailed = 0;
  
  for (const album of albums) {
    console.log(`\nProcessing Album: ${album.title} (ID: ${album.id})`);
    const photos = db.getPhotosByAlbum(album.id);
    
    if (photos.length === 0) {
      console.log('  No photos in this album.');
      continue;
    }
    
    console.log(`  Found ${photos.length} photos`);
    
    for (const photo of photos) {
      totalProcessed++;
      const result = await generateThumbnail(album.id, photo.filename);
      
      if (result) {
        // Check if it was newly generated or already existed
        const thumbPath = path.join(__dirname, '..', 'uploads', String(album.id), 'thumbs', photo.filename);
        const stats = fs.statSync(thumbPath);
        const isNew = Date.now() - stats.mtimeMs < 5000; // Generated in last 5 seconds
        
        if (isNew) {
          totalSuccess++;
        } else {
          totalSkipped++;
        }
      } else {
        totalFailed++;
      }
    }
  }
  
  console.log('\n========================================');
  console.log('Thumbnail Generation Complete!');
  console.log('========================================');
  console.log(`Total photos processed: ${totalProcessed}`);
  console.log(`Newly generated: ${totalSuccess}`);
  console.log(`Already existed: ${totalSkipped}`);
  console.log(`Failed: ${totalFailed}`);
  console.log('========================================\n');
}

// Run the script
generateAllThumbnails()
  .then(() => {
    console.log('Done!');
    process.exit(0);
  })
  .catch((error) => {
    console.error('Fatal error:', error);
    process.exit(1);
  });

