#!/bin/bash
# Setup script for NRW Web Google Cloud project
# Run this AFTER enabling billing on the nrwphotography project

set -e  # Exit on error

PROJECT_ID="nrwphotography"
BUCKET_NAME="nrwphotography-photowebsite"
REGION="europe-west1"

echo "🚀 Setting up NRW Web project..."
echo "Project ID: $PROJECT_ID"
echo "Bucket: $BUCKET_NAME"
echo "Region: $REGION"
echo ""

# Step 1: Enable required APIs
echo "📋 Step 1/5: Enabling required APIs..."
gcloud services enable \
  run.googleapis.com \
  storage.googleapis.com \
  secretmanager.googleapis.com \
  cloudbuild.googleapis.com \
  --project=$PROJECT_ID

echo "✅ APIs enabled"
echo ""

# Step 2: Create Storage bucket
echo "📦 Step 2/5: Creating Cloud Storage bucket..."
gsutil mb -l $REGION gs://$BUCKET_NAME 2>/dev/null || echo "Bucket already exists"
gsutil iam ch allUsers:objectViewer gs://$BUCKET_NAME

echo "✅ Bucket created and made public"
echo ""

# Step 3: Create service account for GitHub Actions
echo "🔐 Step 3/5: Creating GitHub Actions service account..."
gcloud iam service-accounts create github-actions \
  --display-name="GitHub Actions Deployer" \
  --project=$PROJECT_ID 2>/dev/null || echo "Service account already exists"

# Grant permissions
echo "Granting permissions..."
gcloud projects add-iam-policy-binding $PROJECT_ID \
  --member="serviceAccount:github-actions@${PROJECT_ID}.iam.gserviceaccount.com" \
  --role="roles/run.admin" \
  --condition=None

gcloud projects add-iam-policy-binding $PROJECT_ID \
  --member="serviceAccount:github-actions@${PROJECT_ID}.iam.gserviceaccount.com" \
  --role="roles/cloudbuild.builds.editor" \
  --condition=None

gcloud projects add-iam-policy-binding $PROJECT_ID \
  --member="serviceAccount:github-actions@${PROJECT_ID}.iam.gserviceaccount.com" \
  --role="roles/storage.admin" \
  --condition=None

gcloud projects add-iam-policy-binding $PROJECT_ID \
  --member="serviceAccount:github-actions@${PROJECT_ID}.iam.gserviceaccount.com" \
  --role="roles/iam.serviceAccountUser" \
  --condition=None

gcloud projects add-iam-policy-binding $PROJECT_ID \
  --member="serviceAccount:github-actions@${PROJECT_ID}.iam.gserviceaccount.com" \
  --role="roles/viewer" \
  --condition=None

echo "✅ Service account created and configured"
echo ""

# Step 4: Create service account key
echo "🔑 Step 4/5: Creating service account key..."
gcloud iam service-accounts keys create nrw-github-actions-key.json \
  --iam-account=github-actions@${PROJECT_ID}.iam.gserviceaccount.com \
  --project=$PROJECT_ID

echo "✅ Service account key created: nrw-github-actions-key.json"
echo ""

# Step 5: Create admin password secret
echo "🔒 Step 5/5: Creating admin password secret..."
read -sp "Enter admin password for NRW Web: " ADMIN_PASSWORD
echo ""

echo -n "$ADMIN_PASSWORD" | \
  gcloud secrets create photowebsite-admin-password \
  --data-file=- \
  --project=$PROJECT_ID 2>/dev/null || \
  echo "Secret already exists. To update, delete it first."

echo ""
echo "✅ Admin password secret created"
echo ""
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo "✅ NRW Web project setup complete!"
echo "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"
echo ""
echo "📝 NEXT STEPS:"
echo ""
echo "1. Add service account key to GitHub:"
echo "   a. Open: nrw-github-actions-key.json"
echo "   b. Copy the entire JSON content"
echo "   c. Go to: https://github.com/thedoctor0124/photowebsite/settings/secrets/actions"
echo "   d. Click 'New repository secret'"
echo "   e. Name: GCP_SA_KEY_NRW"
echo "   f. Value: Paste the JSON"
echo "   g. Click 'Add secret'"
echo ""
echo "2. IMPORTANT: Delete the key file after adding to GitHub:"
echo "   del nrw-github-actions-key.json"
echo ""
echo "3. Push changes to GitHub to trigger deployment:"
echo "   git push origin main"
echo ""
echo "🎉 Both projects will now deploy automatically!"

