# GitHub Setup & Deployment Workflow

This guide shows you how to set up this repository on GitHub and deploy to multiple Cloud Run instances easily.

## Initial GitHub Setup

### 1. Create GitHub Repository

1. Go to https://github.com/new
2. Name your repository (e.g., `photowebsite`)
3. Make it **private** (recommended for security)
4. Don't initialize with README (we already have one)
5. Click "Create repository"

### 2. Push Code to GitHub

```bash
# Initialize git (if not already done)
git init

# Add all files
git add .

# Make initial commit
git commit -m "Initial commit - Multi-site photo gallery platform"

# Add GitHub as remote (replace with your URL)
git remote add origin https://github.com/YOUR_USERNAME/photowebsite.git

# Push to GitHub
git branch -M main
git push -u origin main
```

### 3. Verify Upload

Go to your GitHub repository and verify all files are there (except those in `.gitignore`).

## Deploying Your First Site

### Step 1: Create GCP Project

```bash
# Set your project ID
PROJECT_ID=photosite-prod
SITE_NAME=site1

# Create project (if needed)
gcloud projects create ${PROJECT_ID}
gcloud config set project ${PROJECT_ID}

# Enable billing (required)
# Visit: https://console.cloud.google.com/billing

# Enable required APIs
gcloud services enable run.googleapis.com
gcloud services enable storage.googleapis.com
gcloud services enable secretmanager.googleapis.com
gcloud services enable cloudbuild.googleapis.com
```

### Step 2: Create Storage Bucket

```bash
BUCKET_NAME=${PROJECT_ID}-${SITE_NAME}

# Create bucket in europe-west1 (free tier)
gsutil mb -l europe-west1 gs://${BUCKET_NAME}

# Make bucket publicly readable
gsutil iam ch allUsers:objectViewer gs://${BUCKET_NAME}
```

### Step 3: Set Admin Password

```bash
# Create secret for admin password
echo -n "your-secure-password-here" | \
  gcloud secrets create photowebsite-admin-password \
  --data-file=- \
  --project=${PROJECT_ID}
```

### Step 4: Clone & Deploy

```bash
# Clone your repo (if not already local)
git clone https://github.com/YOUR_USERNAME/photowebsite.git
cd photowebsite

# Deploy to Cloud Run
gcloud run deploy photowebsite-${SITE_NAME} \
  --source . \
  --platform managed \
  --region europe-west1 \
  --allow-unauthenticated \
  --set-env-vars="USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${BUCKET_NAME}" \
  --set-env-vars="AGE_VERIFICATION_ENABLED=true,PRIMARY_COLOR=#a6ff00,LOGO_URL=/logo.png,SITE_NAME=My Gallery" \
  --set-secrets="ADMIN_PASSWORD=photowebsite-admin-password:latest" \
  --memory=256Mi \
  --cpu=1 \
  --max-instances=10 \
  --min-instances=0 \
  --timeout=300 \
  --project=${PROJECT_ID}
```

### Step 5: Get Your URL

```bash
# Get service URL
gcloud run services describe photowebsite-${SITE_NAME} \
  --region=europe-west1 \
  --format='value(status.url)' \
  --project=${PROJECT_ID}
```

Visit the URL and log in to `/admin` with:
- Username: `admin`
- Password: `<your-secure-password-here>`

## Deploying Additional Sites

To deploy a second site (e.g., different branding):

```bash
# Set site-specific variables
SITE_NAME=portraits
BUCKET_NAME=${PROJECT_ID}-${SITE_NAME}

# Create new bucket
gsutil mb -l europe-west1 gs://${BUCKET_NAME}
gsutil iam ch allUsers:objectViewer gs://${BUCKET_NAME}

# Deploy with different configuration
gcloud run deploy photowebsite-${SITE_NAME} \
  --source . \
  --platform managed \
  --region europe-west1 \
  --allow-unauthenticated \
  --set-env-vars="USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${BUCKET_NAME}" \
  --set-env-vars="AGE_VERIFICATION_ENABLED=false,PRIMARY_COLOR=#3366ff,LOGO_URL=/logo-portraits.png,SITE_NAME=Portrait Studio" \
  --set-secrets="ADMIN_PASSWORD=photowebsite-admin-password:latest" \
  --memory=256Mi \
  --project=${PROJECT_ID}
```

## Updating Sites After Code Changes

When you make changes to the code:

```bash
# Commit and push to GitHub
git add .
git commit -m "Update: feature description"
git push origin main

# Pull latest on your deployment machine
git pull origin main

# Redeploy all sites (they'll get the latest code)
gcloud run deploy photowebsite-site1 --source . --region europe-west1 --project=${PROJECT_ID}
gcloud run deploy photowebsite-portraits --source . --region europe-west1 --project=${PROJECT_ID}
```

## Automated Deployment with Cloud Build

For automatic deployment on every GitHub push:

### 1. Connect GitHub to Cloud Build

```bash
# Visit Cloud Build triggers page
gcloud builds triggers create github \
  --repo-name=photowebsite \
  --repo-owner=YOUR_USERNAME \
  --branch-pattern="^main$" \
  --build-config=cloudbuild.yaml \
  --project=${PROJECT_ID}
```

### 2. Update Substitutions

Edit `cloudbuild.yaml` to match your site's configuration:

```yaml
substitutions:
  _DEPLOY_REGION: 'europe-west1'
  _GCS_BUCKET_NAME: 'your-bucket-name'
  _AGE_VERIFICATION_ENABLED: 'true'
  _PRIMARY_COLOR: '#a6ff00'
  _LOGO_URL: '/logo.png'
  _SITE_NAME: 'Your Site Name'
```

### 3. Push to Deploy

```bash
git add cloudbuild.yaml
git commit -m "Configure Cloud Build for site1"
git push origin main
```

Now every push to `main` will automatically deploy to Cloud Run!

## Managing Multiple Sites with Multiple Repos

If you want completely separate configs for each site:

### Option 1: Branches

```bash
# Create branch for each site
git checkout -b site1-config
# Edit cloudbuild.yaml with site1 settings
git commit -am "Site 1 configuration"
git push origin site1-config

git checkout -b site2-config
# Edit cloudbuild.yaml with site2 settings
git commit -am "Site 2 configuration"
git push origin site2-config
```

Create separate Cloud Build triggers for each branch.

### Option 2: Separate Config Files

```bash
# Create multiple Cloud Build configs
cp cloudbuild.yaml cloudbuild-site1.yaml
cp cloudbuild.yaml cloudbuild-site2.yaml

# Edit each with site-specific substitutions
# Deploy manually with:
gcloud builds submit --config=cloudbuild-site1.yaml
gcloud builds submit --config=cloudbuild-site2.yaml
```

## Custom Domains

To add a custom domain to any site:

```bash
# Map domain
gcloud run domain-mappings create \
  --service photowebsite-site1 \
  --domain www.yoursite.com \
  --region europe-west1 \
  --project=${PROJECT_ID}

# Get DNS records
gcloud run domain-mappings describe \
  --domain www.yoursite.com \
  --region europe-west1 \
  --project=${PROJECT_ID}
```

Add the provided DNS records to your domain registrar.

## Configuration Reference

### Environment Variables

| Variable | Required | Default | Description |
|----------|----------|---------|-------------|
| `ADMIN_PASSWORD` | Yes | `changeme` | Admin panel password (use Secret Manager) |
| `USE_JSON` | Yes | `true` | Use JSON database (vs SQLite) |
| `USE_CLOUD_STORAGE` | Yes | `true` | Store photos in GCS (vs local) |
| `GCS_BUCKET_NAME` | Yes | - | GCS bucket name for photos |
| `AGE_VERIFICATION_ENABLED` | No | `true` | Show 18+ age gate |
| `PRIMARY_COLOR` | No | `#a6ff00` | Brand color (hex) |
| `LOGO_URL` | No | `/logo.png` | Logo path/URL |
| `SITE_NAME` | No | `Lucky Pup Photography` | Site name |

### Customizing Logo

**Option 1: Commit to repo**
```bash
# Add logo to public directory
cp /path/to/your-logo.png public/logo-yoursite.png

# Set LOGO_URL
--set-env-vars="LOGO_URL=/logo-yoursite.png"

# Commit and deploy
git add public/logo-yoursite.png
git commit -m "Add custom logo"
git push
```

**Option 2: Upload to bucket**
```bash
# Upload logo to bucket
gsutil cp /path/to/logo.png gs://${BUCKET_NAME}/logo.png

# Set LOGO_URL to GCS URL
--set-env-vars="LOGO_URL=https://storage.googleapis.com/${BUCKET_NAME}/logo.png"
```

### Changing Configuration

Update environment variables without redeploying code:

```bash
gcloud run services update photowebsite-site1 \
  --update-env-vars="PRIMARY_COLOR=#ff6b00,SITE_NAME=New Name" \
  --region europe-west1 \
  --project=${PROJECT_ID}
```

## Security Best Practices

1. **Keep repo private** - Contains site structure
2. **Use Secret Manager** - Never commit passwords
3. **Restrict admin access** - Use strong passwords
4. **Enable Cloud Armor** (optional) - Rate limiting/DDoS protection
5. **Monitor logs** - Check for suspicious activity

## Troubleshooting

### Deployment fails

```bash
# Check logs
gcloud run services logs read photowebsite-site1 --region=europe-west1

# Check build logs
gcloud builds list --project=${PROJECT_ID}
```

### Can't access admin panel

1. Check username is `admin` (lowercase)
2. Verify password matches Secret Manager
3. Redeploy service after password change

### Images not showing

1. Check bucket permissions: `gsutil iam ch allUsers:objectViewer gs://${BUCKET_NAME}`
2. Verify `GCS_BUCKET_NAME` matches actual bucket
3. Check Cloud Run logs for upload errors

### Age gate not showing/hiding

- Set `AGE_VERIFICATION_ENABLED=false` to disable
- Clear cookies to reset age verification

## Cost Optimization

- **Use single GCP project** for all sites (share free tier)
- **Set min-instances=0** (no idle cost)
- **Use small memory** (256Mi sufficient)
- **Enable CDN** for Cloud Storage (optional, better performance)
- **Monitor usage** to stay in free tier

## Next Steps

1. Set up Cloud Build triggers for automatic deployment
2. Configure custom domains
3. Add Google Analytics (optional)
4. Set up monitoring alerts
5. Create backup scripts for JSON data

---

Need help? Open an issue on GitHub or check the [main README](README.md).

