/**
 * AI Assistant Console JavaScript
 * 
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

(function() {
    'use strict';

    const AIConsole = {
        currentSession: null,

        init() {
            this.bindEvents();
            console.log('AI Assistant Console initialized');
        },

        bindEvents() {
            // Submit prompt
            document.getElementById('submit-btn').addEventListener('click', () => {
                this.submitPrompt();
            });

            // Enter key in textarea (Ctrl+Enter to submit)
            document.getElementById('prompt-input').addEventListener('keydown', (e) => {
                if (e.ctrlKey && e.key === 'Enter') {
                    this.submitPrompt();
                }
            });

            // History button
            document.getElementById('history-btn').addEventListener('click', () => {
                this.showHistory();
            });

            // Close history
            document.getElementById('close-history').addEventListener('click', () => {
                this.hideHistory();
            });

            // Test API button
            document.getElementById('test-api-btn').addEventListener('click', () => {
                this.testApiConnection();
            });

            // View Logs button
            document.getElementById('view-logs-btn').addEventListener('click', () => {
                this.viewLogs();
            });

            // Settings button
            const settingsBtn = document.getElementById('settings-btn');
            if (settingsBtn) {
                settingsBtn.addEventListener('click', () => {
                    this.showSettings();
                });
            }

            // Close review modal
            document.getElementById('close-review').addEventListener('click', () => {
                this.hideReviewModal();
            });

            // Execute actions
            document.getElementById('execute-all-btn').addEventListener('click', () => {
                this.executeActions();
            });

            // Cancel review
            document.getElementById('cancel-review-btn').addEventListener('click', () => {
                this.hideReviewModal();
            });
        },

        async submitPrompt() {
            const promptInput = document.getElementById('prompt-input');
            const prompt = promptInput.value.trim();
            
            if (!prompt) {
                alert('Please enter a prompt');
                return;
            }

            // Always auto-execute read actions, only review write actions if checked
            const requireReview = document.getElementById('require-review').checked;

            // Add user message to conversation
            this.addMessage('user', prompt);
            
            // Clear input
            promptInput.value = '';

            // Show immediate feedback before AI starts
            const thinkingPlaceholder = document.createElement('div');
            thinkingPlaceholder.className = 'ai-thinking-placeholder';
            thinkingPlaceholder.innerHTML = `
                <div class="thinking-animation">
                    <div class="thinking-dot"></div>
                    <div class="thinking-dot"></div>
                    <div class="thinking-dot"></div>
                </div>
                <p>🔍 Analyzing your request...</p>
            `;
            const conversation = document.getElementById('conversation');
            if (conversation) {
                conversation.appendChild(thinkingPlaceholder);
                this.scrollToBottom();
            }

            try {
                // Use SSE for real-time streaming updates
                await this.streamingRequest(prompt, requireReview);
                
                // Remove placeholder if still there (in case of error)
                if (thinkingPlaceholder && thinkingPlaceholder.parentNode) {
                    thinkingPlaceholder.remove();
                }

            } catch (error) {
                // Remove placeholder on error
                if (thinkingPlaceholder && thinkingPlaceholder.parentNode) {
                    thinkingPlaceholder.remove();
                }
                this.hideLoading();
                this.addMessage('error', 'Failed to process prompt: ' + (error.message || String(error)));
            }
        },

        // Real-time streaming request handler (SSE)
        async streamingRequest(prompt, requireReview) {
            const url = window.location.href.split('?')[0] + 
                '?option=com_ajax&plugin=aiassistant&group=system&format=json&task=process';
            
            // Include CSRF token in the body
            const body = JSON.stringify({
                prompt: prompt,
                require_review: requireReview,
                streaming: true,
                [window.AI_ASSISTANT_CONFIG.token]: 1
            });

            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: body
            });

            if (!response.ok) {
                // Try to get error details
                const contentType = response.headers.get('content-type');
                if (contentType && contentType.includes('application/json')) {
                    const errorData = await response.json();
                    throw new Error(errorData.error || 'HTTP error ' + response.status);
                }
                throw new Error('HTTP error ' + response.status);
            }

            // Check if we got SSE response
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('text/event-stream')) {
                console.warn('Expected SSE response but got:', contentType);
                // Fall back to JSON response
                if (contentType && contentType.includes('application/json')) {
                    const jsonResponse = await response.json();
                    if (jsonResponse.error) {
                        throw new Error(jsonResponse.error);
                    }
                    // Handle non-streaming response
                    this.currentSession = jsonResponse;
                    if (jsonResponse.ai_response) {
                        this.addMessage('assistant', jsonResponse.ai_response);
                    }
                    this.displayResult(jsonResponse);
                    return;
                }
                throw new Error('Unexpected response type. Expected streaming response but got: ' + contentType);
            }

            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            let buffer = '';
            let currentIterationDiv = null;
            let iterationThoughts = [];

            // Process the stream
            while (true) {
                const { done, value } = await reader.read();
                
                if (done) {
                    this.hideLoading();
                    break;
                }

                buffer += decoder.decode(value, { stream: true });
                const lines = buffer.split('\n\n');
                buffer = lines.pop(); // Keep incomplete line in buffer

                for (const line of lines) {
                    if (!line.trim()) continue;
                    
                    // Parse SSE format: "event: type\ndata: json"
                    const eventMatch = line.match(/event:\s*(\w+)\s*\ndata:\s*(.+)/s);
                    if (!eventMatch) continue;

                    const eventType = eventMatch[1];
                    let eventData;
                    
                    try {
                        eventData = JSON.parse(eventMatch[2]);
                    } catch (e) {
                        console.error('Failed to parse SSE data:', eventMatch[2], e);
                        this.addMessage('error', 'Failed to parse server response. The server may have returned an error. Check browser console for details.');
                        continue;
                    }

                    // Handle different event types
                    switch (eventType) {
                        case 'iteration_start':
                            // Remove placeholder when first iteration starts
                            const placeholder = document.querySelector('.ai-thinking-placeholder');
                            if (placeholder) {
                                placeholder.remove();
                            }
                            
                            // Create new iteration display
                            if (!currentIterationDiv) {
                                currentIterationDiv = document.createElement('div');
                                currentIterationDiv.className = 'ai-thinking-process';
                                currentIterationDiv.innerHTML = '<h4>🧠 Agent Reasoning:</h4>';
                                const conversation = document.getElementById('conversation');
                                if (conversation) {
                                    conversation.appendChild(currentIterationDiv);
                                }
                            }
                            
                            const thoughtItem = document.createElement('div');
                            thoughtItem.className = 'thought-iteration';
                            thoughtItem.innerHTML = `
                                <div class="thought-header">💭 Iteration ${eventData.data.iteration}</div>
                                <div class="thought-content">${this.escapeHtml(eventData.data.thought)}</div>
                                <div class="thought-actions">⏳ Working...</div>
                            `;
                            currentIterationDiv.appendChild(thoughtItem);
                            this.scrollToBottom();
                            break;

                        case 'iteration_update':
                            // Update the thought content with the real AI thought
                            if (currentIterationDiv) {
                                const targetIteration = currentIterationDiv.querySelector(`.thought-iteration:nth-child(${eventData.data.iteration + 1}) .thought-content`);
                                if (targetIteration) {
                                    targetIteration.textContent = eventData.data.thought;
                                }
                            }
                            break;

                        case 'action_result':
                            // Update last iteration to show it's done
                            if (currentIterationDiv) {
                                const lastThought = currentIterationDiv.querySelector('.thought-iteration:last-child .thought-actions');
                                if (lastThought) {
                                    lastThought.textContent = '✅ Completed';
                                }
                            }
                            break;

                        case 'complete':
                            // Final result received
                            this.hideLoading();
                            this.currentSession = eventData.data;

                            // Show AI's final answer
                            if (eventData.data.ai_response && eventData.data.ai_response.trim()) {
                                this.addMessage('assistant', eventData.data.ai_response);
                            }

                            // Display final result summary
                            this.displayResult(eventData.data);

                            // If review is required, show review modal
                            if (requireReview && eventData.data.status === 'pending_review') {
                                this.showReviewModal(eventData.data);
                            }
                            break;

                        case 'error':
                            this.hideLoading();
                            this.addMessage('error', eventData.data.message);
                            break;
                    }
                }
            }
        },

        displayResult(result) {
            const summary = result.summary || 'Task completed';
            const actions = result.actions || [];
            const iterationThoughts = result.iteration_thoughts || [];

            // If there are no actions and no AI response was already shown, don't show anything
            if (actions.length === 0 && (!result.ai_response || !result.ai_response.trim())) {
                return;
            }

            // Create a container div
            const resultDiv = document.createElement('div');
            resultDiv.className = 'ai-result';

            // Show iterative thinking process (like Cursor Composer)
            if (iterationThoughts.length > 0) {
                const thinkingDiv = document.createElement('div');
                thinkingDiv.className = 'ai-thinking-process';
                thinkingDiv.innerHTML = '<h4>🧠 Agent Reasoning:</h4>';
                
                iterationThoughts.forEach(item => {
                    const thoughtItem = document.createElement('div');
                    thoughtItem.className = 'thought-iteration';
                    thoughtItem.innerHTML = `
                        <div class="thought-header">💭 Iteration ${item.iteration}</div>
                        <div class="thought-content">${this.escapeHtml(item.thought)}</div>
                        <div class="thought-actions">→ Requested ${item.action_count} action(s)</div>
                    `;
                    thinkingDiv.appendChild(thoughtItem);
                });
                
                resultDiv.appendChild(thinkingDiv);
            }

            // Add summary only if there are actions
            if (actions.length > 0) {
                const summaryDiv = document.createElement('div');
                summaryDiv.className = 'ai-result-summary';
                summaryDiv.textContent = summary;
                resultDiv.appendChild(summaryDiv);
            }

            if (actions.length > 0) {
                const actionsDiv = document.createElement('div');
                actionsDiv.className = 'ai-result-actions';
                
                const heading = document.createElement('h4');
                heading.textContent = `Actions (${actions.length}):`;
                actionsDiv.appendChild(heading);
                
                const ul = document.createElement('ul');
                
                for (const action of actions) {
                    const li = document.createElement('li');
                    li.className = `action-item ${this.getStatusClass(action.status)}`;
                    
                    const statusIcon = document.createTextNode(this.getStatusIcon(action.status) + ' ');
                    const actionType = document.createElement('strong');
                    actionType.textContent = action.type;
                    
                    li.appendChild(statusIcon);
                    li.appendChild(actionType);
                    
                    if (action.reason) {
                        li.appendChild(document.createTextNode(' - ' + action.reason));
                    }
                    
                    // Add undo button for executed actions
                    if (action.status === 'executed' && action.id && action.type !== 'undo_action') {
                        const undoBtn = document.createElement('button');
                        undoBtn.className = 'btn-undo';
                        undoBtn.textContent = '↶ Undo';
                        undoBtn.title = 'Undo this action';
                        undoBtn.onclick = () => this.undoAction(action.id);
                        li.appendChild(document.createTextNode(' '));
                        li.appendChild(undoBtn);
                    }
                    
                    if (action.status === 'executed' && action.result) {
                        const resultDiv = document.createElement('div');
                        resultDiv.className = 'action-result';
                        const resultPre = document.createElement('pre');
                        resultPre.textContent = typeof action.result === 'object' 
                            ? JSON.stringify(action.result, null, 2) 
                            : String(action.result);
                        resultDiv.appendChild(resultPre);
                        li.appendChild(resultDiv);
                    }
                    
                    if (action.status === 'failed' && action.error) {
                        const errorDiv = document.createElement('div');
                        errorDiv.className = 'action-error';
                        errorDiv.textContent = action.error;
                        li.appendChild(errorDiv);
                    }
                    
                    if (action.status === 'undone') {
                        const undoneDiv = document.createElement('div');
                        undoneDiv.className = 'action-undone';
                        undoneDiv.textContent = '✓ This action has been undone';
                        li.appendChild(undoneDiv);
                    }
                    
                    ul.appendChild(li);
                }
                
                actionsDiv.appendChild(ul);
                resultDiv.appendChild(actionsDiv);
            }

            // Only add the result message if there's content to show
            if (resultDiv.children.length > 0) {
                this.addMessage('assistant', resultDiv.outerHTML, true);
            }
        },

        showReviewModal(result) {
            const modal = document.getElementById('review-modal');
            const actionsContainer = document.getElementById('review-actions');

            let html = `<div class="ai-review-info">`;
            html += `<p><strong>Session:</strong> ${result.session_id}</p>`;
            html += `<p>The AI has planned ${result.actions.length} action(s). Review them and choose to execute:</p>`;
            html += `</div>`;

            html += `<div class="ai-review-actions">`;
            
            for (let i = 0; i < result.actions.length; i++) {
                const action = result.actions[i];
                html += `<div class="review-action-item">`;
                html += `<h4>${i + 1}. ${this.escapeHtml(action.type)}</h4>`;
                html += `<p class="action-reason">${this.escapeHtml(action.reason || '')}</p>`;
                html += `<div class="action-params">`;
                html += `<strong>Parameters:</strong>`;
                html += `<pre>${JSON.stringify(action.parameters, null, 2)}</pre>`;
                html += `</div>`;
                html += `</div>`;
            }
            
            html += `</div>`;

            actionsContainer.innerHTML = html;
            modal.style.display = 'flex';
        },

        hideReviewModal() {
            document.getElementById('review-modal').style.display = 'none';
        },

        async executeActions() {
            if (!this.currentSession || !this.currentSession.session_id) {
                alert('No session to execute');
                return;
            }

            this.hideReviewModal();
            this.showLoading('Executing actions...');

            try {
                const response = await this.apiRequest('execute', {
                    session_id: this.currentSession.session_id
                });

                this.hideLoading();

                if (response.error) {
                    this.addMessage('error', response.error);
                    return;
                }

                this.addMessage('assistant', 
                    `Actions executed successfully! ${response.summary}`);

            } catch (error) {
                this.hideLoading();
                this.addMessage('error', 'Failed to execute actions: ' + error.message);
            }
        },

        async showHistory() {
            const sidebar = document.getElementById('history-sidebar');
            const listContainer = document.getElementById('history-list');

            sidebar.classList.add('active');
            listContainer.innerHTML = '<div class="ai-loading">Loading history...</div>';

            try {
                // Build URL for GET request with proper format
                const baseUrl = window.AI_ASSISTANT_CONFIG.baseUrl;
                const adminPath = baseUrl.includes('/administrator/') ? '' : 'administrator/';
                const url = baseUrl + adminPath + 
                            'index.php?option=com_ajax&plugin=aiassistant&group=system&format=json&task=history';

                const response = await fetch(url, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json'
                    }
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }

                const data = await response.json();

                if (data.error) {
                    listContainer.innerHTML = `<div class="ai-error">${data.error}</div>`;
                    return;
                }

                const sessions = data.sessions || [];

                if (sessions.length === 0) {
                    listContainer.innerHTML = '<div class="ai-empty">No history yet</div>';
                    return;
                }

                let html = '<div class="history-list">';
                
                for (const session of sessions) {
                    const date = new Date(session.started_at);
                    const dateStr = date.toLocaleString();
                    const statusClass = this.getStatusClass(session.status);
                    
                    html += `<div class="history-item ${statusClass}">`;
                    html += `<div class="history-prompt">${this.escapeHtml(session.prompt.substring(0, 100))}...</div>`;
                    html += `<div class="history-meta">`;
                    html += `<span class="history-date">${dateStr}</span>`;
                    html += `<span class="history-status">${session.status}</span>`;
                    html += `</div>`;
                    html += `</div>`;
                }
                
                html += '</div>';

                listContainer.innerHTML = html;

            } catch (error) {
                console.error('History error:', error);
                listContainer.innerHTML = `<div class="ai-error">Failed to load history: ${error.message}</div>`;
            }
        },

        hideHistory() {
            document.getElementById('history-sidebar').classList.remove('active');
        },

        async testApiConnection() {
            this.showLoading('Testing API connection...');

            try {
                const response = await this.apiRequest('test_api', {}, 'GET');
                this.hideLoading();

                let message = `<div class="ai-diagnostic">`;
                message += `<h3>API Configuration Test</h3>`;
                message += `<p><strong>Provider:</strong> ${response.provider}</p>`;
                message += `<p><strong>Model:</strong> ${response.model}</p>`;
                message += `<p><strong>API Key Configured:</strong> ${response.api_key_configured ? '✓ Yes' : '✗ No'}</p>`;
                message += `<p><strong>API Key Length:</strong> ${response.api_key_length} characters</p>`;
                message += `<p><strong>API Key Starts:</strong> ${response.api_key_starts_with}</p>`;
                
                if (response.issues && response.issues.length > 0) {
                    message += `<h4 style="color: orange;">⚠ Issues Detected:</h4><ul>`;
                    response.issues.forEach(issue => {
                        message += `<li>${this.escapeHtml(issue)}</li>`;
                    });
                    message += `</ul>`;
                } else {
                    message += `<p style="color: green;"><strong>✓ ${response.status}</strong></p>`;
                }
                
                message += `</div>`;
                this.addMessage('assistant', message, true);

            } catch (error) {
                this.hideLoading();
                this.addMessage('error', 'Failed to test API: ' + error.message);
            }
        },

        async viewLogs() {
            this.showLoading('Loading logs...');

            try {
                const response = await this.apiRequest('view_logs', {}, 'GET');
                this.hideLoading();

                if (!response.success) {
                    this.addMessage('error', response.message || 'Failed to load logs');
                    return;
                }

                let message = `<div class="ai-logs">`;
                message += `<h3>Recent Logs (Last ${response.log_count} entries)</h3>`;
                message += `<p style="font-size: 12px; color: #666;">Log file: ${response.log_path}</p>`;
                message += `<div style="background: #000; color: #0f0; padding: 10px; max-height: 400px; overflow-y: auto; font-family: monospace; font-size: 11px;">`;
                
                response.logs.forEach(log => {
                    let color = '#0f0';
                    if (log.includes('[ERROR]') || log.includes('[FATAL]')) {
                        color = '#f00';
                    } else if (log.includes('[WARN]')) {
                        color = '#ff0';
                    } else if (log.includes('[DEBUG]')) {
                        color = '#0ff';
                    }
                    message += `<div style="color: ${color};">${this.escapeHtml(log)}</div>`;
                });
                
                message += `</div></div>`;
                this.addMessage('assistant', message, true);

            } catch (error) {
                this.hideLoading();
                this.addMessage('error', 'Failed to load logs: ' + error.message);
            }
        },

        showSettings() {
            // Open plugin settings in new tab
            const baseUrl = window.AI_ASSISTANT_CONFIG.baseUrl;
            const adminPath = baseUrl.includes('/administrator/') ? '' : 'administrator/';
            const settingsUrl = baseUrl + adminPath + 'index.php?option=com_plugins&view=plugin&layout=edit&extension_id=';
            
            // Try to find the plugin ID
            alert('Settings:\n\nTo configure the AI Assistant plugin, go to:\nSystem → Plugins → System - AI Assistant\n\nOr close this dialog and we\'ll try to open it for you.');
            
            // Try to open settings (may need plugin ID)
            window.open(baseUrl + adminPath + 'index.php?option=com_plugins&filter[folder]=system&filter[element]=aiassistant', '_blank');
        },

        addMessage(role, content, isHtml = false) {
            const conversation = document.getElementById('conversation');
            const messageDiv = document.createElement('div');
            messageDiv.className = `ai-message ai-message-${role}`;

            const contentDiv = document.createElement('div');
            contentDiv.className = 'ai-message-content';

            if (isHtml) {
                contentDiv.innerHTML = content;
            } else {
                contentDiv.textContent = content;
            }

            messageDiv.appendChild(contentDiv);
            conversation.appendChild(messageDiv);

            // Scroll to bottom
            conversation.scrollTop = conversation.scrollHeight;
        },

        showLoading(message = 'AI is thinking...') {
            const overlay = document.getElementById('loading-overlay');
            overlay.querySelector('p').textContent = message;
            overlay.style.display = 'flex';
        },

        hideLoading() {
            document.getElementById('loading-overlay').style.display = 'none';
        },

        async apiRequest(task, data = {}, method = 'POST') {
            try {
                // Build proper administrator URL
                const baseUrl = window.AI_ASSISTANT_CONFIG.baseUrl;
                const adminPath = baseUrl.includes('/administrator/') ? '' : 'administrator/';
                const url = baseUrl + adminPath + 
                            'index.php?option=com_ajax&plugin=aiassistant&group=system&format=json&task=' + 
                            encodeURIComponent(task);

                const options = {
                    method: method,
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    // Increase timeout for long AI responses (browser will wait up to 5 minutes)
                    signal: AbortSignal.timeout ? AbortSignal.timeout(300000) : undefined
                };

                if (method === 'POST') {
                    data[window.AI_ASSISTANT_CONFIG.token] = 1;
                    options.body = JSON.stringify(data);
                }

                const response = await fetch(url, options);
                
                if (!response.ok) {
                    // Try to get error message from response
                    let errorMsg = `HTTP ${response.status}: ${response.statusText}`;
                    try {
                        const errorData = await response.json();
                        if (errorData.error) {
                            errorMsg = errorData.error;
                        }
                        console.error('Server error response:', errorData);
                    } catch (e) {
                        console.error('Could not parse error response');
                    }
                    throw new Error(errorMsg);
                }

                // Check if response is JSON
                const contentType = response.headers.get('content-type');
                if (!contentType || !contentType.includes('application/json')) {
                    const text = await response.text();
                    console.error('Non-JSON response received:', text);
                    throw new Error('Server returned non-JSON response. Please check if the plugin is properly configured.');
                }

                return await response.json();
            } catch (error) {
                console.error('API Request Error:', error);
                throw error;
            }
        },

        getStatusClass(status) {
            const map = {
                'executed': 'status-success',
                'completed': 'status-success',
                'pending_review': 'status-warning',
                'failed': 'status-error',
                'error': 'status-error'
            };
            return map[status] || 'status-info';
        },

        getStatusIcon(status) {
            const map = {
                'executed': '✓',
                'completed': '✓',
                'pending_review': '⏳',
                'failed': '✗',
                'error': '✗',
                'planned': '📝'
            };
            return map[status] || '•';
        },

        formatResult(result) {
            if (typeof result === 'object') {
                // Escape the JSON string to prevent XSS
                const jsonStr = this.escapeHtml(JSON.stringify(result, null, 2));
                return '<pre>' + jsonStr + '</pre>';
            }
            return this.escapeHtml(String(result));
        },

        escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },

        async undoAction(actionId) {
            if (!confirm('Are you sure you want to undo this action? This will restore the previous state.')) {
                return;
            }

            this.showLoading('Undoing action...');

            try {
                const response = await this.apiRequest('process', {
                    prompt: `Undo the action with ID ${actionId}. Use the undo_action with action_id parameter.`,
                    require_review: false
                });

                this.hideLoading();

                if (response.error) {
                    this.addMessage('error', response.error);
                    return;
                }

                this.addMessage('assistant', `Action #${actionId} has been successfully undone!`);
                
                // Refresh to show updated state
                if (this.currentSession) {
                    this.displayResult(response);
                }

            } catch (error) {
                this.hideLoading();
                this.addMessage('error', 'Failed to undo action: ' + error.message);
            }
        },

        scrollToBottom() {
            const conversation = document.getElementById('conversation');
            if (conversation) {
                conversation.scrollTop = conversation.scrollHeight;
            }
        }
    };

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => AIConsole.init());
    } else {
        AIConsole.init();
    }

    // Expose to window for debugging
    window.AIConsole = AIConsole;
})();

