<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Controller;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Plugin\System\AiAssistant\AI\OpenAIProvider;
use Joomla\Plugin\System\AiAssistant\AI\AnthropicProvider;
use Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator;
use Joomla\Plugin\System\AiAssistant\Actions\ActionRegistry;
use Joomla\Plugin\System\AiAssistant\Logger\ActionLogger;

defined('_JEXEC') or die;

/**
 * Console Controller
 * Handles AJAX requests from the AI Console UI
 *
 * @since  1.0.0
 */
class ConsoleController extends BaseController
{
    /**
     * Handle AJAX request
     *
     * @return  void
     * @since   1.0.0
     */
    public function execute()
    {
        $app = Factory::getApplication();
        $input = $app->input;
        $action = $input->get('task', 'display');

        // Check permissions
        $user = $app->getIdentity();
        if (!$user->authorise('core.admin', 'com_plugins')) {
            $this->sendResponse(['error' => 'Unauthorized'], 403);
            return;
        }

        // Route to appropriate method
        switch ($action) {
            case 'process':
                $this->processPrompt();
                break;
            
            case 'execute':
                $this->executeActions();
                break;
            
            case 'history':
                $this->getHistory();
                break;
            
            case 'session':
                $this->getSession();
                break;
            
            case 'display':
            default:
                $this->displayConsole();
                break;
        }
    }

    /**
     * Display the console interface
     *
     * @return  void
     * @since   1.0.0
     */
    private function displayConsole(): void
    {
        $app = Factory::getApplication();
        $doc = $app->getDocument();

        // Load assets
        $doc->addStyleSheet('plugins/system/aiassistant/media/css/console.css');
        $doc->addScript('plugins/system/aiassistant/media/js/console.js');

        // Get plugin params for frontend
        $plugin = PluginHelper::getPlugin('system', 'aiassistant');
        $params = json_decode($plugin->params);

        // Include the console template
        $templatePath = JPATH_PLUGINS . '/system/aiassistant/tmpl/console.php';
        
        if (file_exists($templatePath)) {
            include $templatePath;
        } else {
            echo '<h1>AI Assistant Console</h1><p>Template not found.</p>';
        }
    }

    /**
     * Process a user prompt
     *
     * @return  void
     * @since   1.0.0
     */
    private function processPrompt(): void
    {
        $app = Factory::getApplication();
        $input = $app->input;

        $prompt = $input->post->get('prompt', '', 'raw');
        $requireReview = $input->post->get('require_review', true, 'bool');

        if (empty($prompt)) {
            $this->sendResponse(['error' => 'Prompt is required'], 400);
            return;
        }

        try {
            $orchestrator = $this->getOrchestrator();
            $result = $orchestrator->processPrompt($prompt, $requireReview);
            
            $this->sendResponse($result);

        } catch (\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Execute reviewed actions
     *
     * @return  void
     * @since   1.0.0
     */
    private function executeActions(): void
    {
        $app = Factory::getApplication();
        $input = $app->input;

        $sessionId = $input->post->get('session_id', '', 'string');
        $actionIds = $input->post->get('action_ids', [], 'array');

        if (empty($sessionId)) {
            $this->sendResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $orchestrator = $this->getOrchestrator();
            $result = $orchestrator->executeReviewedActions($sessionId, $actionIds);
            
            $this->sendResponse($result);

        } catch (\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get session history
     *
     * @return  void
     * @since   1.0.0
     */
    private function getHistory(): void
    {
        try {
            $plugin = PluginHelper::getPlugin('system', 'aiassistant');
            $params = json_decode($plugin->params, true);
            
            $logger = new ActionLogger($params['log_all_actions'] ?? true);
            $sessions = $logger->getRecentSessions(20);
            
            $this->sendResponse(['sessions' => $sessions]);

        } catch (\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get specific session details
     *
     * @return  void
     * @since   1.0.0
     */
    private function getSession(): void
    {
        $app = Factory::getApplication();
        $input = $app->input;

        $sessionId = $input->get('session_id', '', 'string');

        if (empty($sessionId)) {
            $this->sendResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $plugin = PluginHelper::getPlugin('system', 'aiassistant');
            $params = json_decode($plugin->params, true);
            
            $logger = new ActionLogger($params['log_all_actions'] ?? true);
            $actions = $logger->getSessionActions($sessionId);
            
            $this->sendResponse(['actions' => $actions]);

        } catch (\Exception $e) {
            $this->sendResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get orchestrator instance
     *
     * @return  AgentOrchestrator
     * @since   1.0.0
     */
    private function getOrchestrator(): AgentOrchestrator
    {
        $plugin = PluginHelper::getPlugin('system', 'aiassistant');
        $params = json_decode($plugin->params, true);

        // Initialize AI provider
        $provider = $params['ai_provider'] ?? 'openai';
        
        if ($provider === 'anthropic') {
            $apiKey = $params['anthropic_api_key'] ?? '';
            $model = $params['anthropic_model'] ?? 'claude-3-5-sonnet-20241022';
            $aiProvider = new AnthropicProvider($apiKey, $model);
        } else {
            $apiKey = $params['openai_api_key'] ?? '';
            $model = $params['openai_model'] ?? 'gpt-4o';
            $aiProvider = new OpenAIProvider($apiKey, $model);
        }

        // Initialize other components
        $actionRegistry = new ActionRegistry($params);
        $logger = new ActionLogger($params['log_all_actions'] ?? true);

        return new AgentOrchestrator($aiProvider, $actionRegistry, $logger, $params);
    }

    /**
     * Send JSON response
     *
     * @param   array  $data        Response data
     * @param   int    $statusCode  HTTP status code
     *
     * @return  void
     * @since   1.0.0
     */
    private function sendResponse(array $data, int $statusCode = 200): void
    {
        $app = Factory::getApplication();
        
        $app->setHeader('Content-Type', 'application/json', true);
        $app->setHeader('Status', $statusCode, true);
        
        echo json_encode($data, JSON_PRETTY_PRINT);
        
        $app->close();
    }
}

