/**
 * Usage limits configuration for different subscription tiers
 * Set via environment variable: SUBSCRIPTION_TIER=starter|professional|business|enterprise
 */

const TIERS = {
  starter: {
    name: 'Starter',
    maxPhotos: 5000,
    maxStorageBytes: 50 * 1024 * 1024 * 1024, // 50GB
    maxBandwidthBytes: 100 * 1024 * 1024 * 1024, // 100GB/month
    maxAlbums: 10,
    maxFileSize: 20 * 1024 * 1024, // 20MB per file
  },
  professional: {
    name: 'Professional',
    maxPhotos: 20000,
    maxStorageBytes: 150 * 1024 * 1024 * 1024, // 150GB
    maxBandwidthBytes: 300 * 1024 * 1024 * 1024, // 300GB/month
    maxAlbums: 50,
    maxFileSize: 30 * 1024 * 1024, // 30MB per file
  },
  business: {
    name: 'Business',
    maxPhotos: 100000,
    maxStorageBytes: 500 * 1024 * 1024 * 1024, // 500GB
    maxBandwidthBytes: 1000 * 1024 * 1024 * 1024, // 1TB/month
    maxAlbums: 9999, // Unlimited
    maxFileSize: 50 * 1024 * 1024, // 50MB per file
  },
  enterprise: {
    name: 'Enterprise',
    maxPhotos: 999999, // Virtually unlimited
    maxStorageBytes: 2000 * 1024 * 1024 * 1024, // 2TB
    maxBandwidthBytes: 5000 * 1024 * 1024 * 1024, // 5TB/month
    maxAlbums: 99999, // Unlimited
    maxFileSize: 100 * 1024 * 1024, // 100MB per file
  }
};

/**
 * Get current tier configuration from environment
 */
function getCurrentTier() {
  const tierName = (process.env.SUBSCRIPTION_TIER || 'starter').toLowerCase();
  const tier = TIERS[tierName];
  
  if (!tier) {
    console.warn(`[LIMITS] Unknown tier "${tierName}", defaulting to "starter"`);
    return { ...TIERS.starter, tier: 'starter' };
  }
  
  return { ...tier, tier: tierName };
}

/**
 * Format bytes to human-readable string
 */
function formatBytes(bytes) {
  if (bytes === 0) return '0 Bytes';
  const k = 1024;
  const sizes = ['Bytes', 'KB', 'MB', 'GB', 'TB'];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return Math.round((bytes / Math.pow(k, i)) * 100) / 100 + ' ' + sizes[i];
}

/**
 * Calculate percentage of limit used
 */
function calculateUsagePercent(used, max) {
  if (max === 0) return 0;
  return Math.round((used / max) * 100);
}

module.exports = {
  TIERS,
  getCurrentTier,
  formatBytes,
  calculateUsagePercent
};

