# NRW Web - Multi-Site Photo Gallery Platform

A serverless photo gallery platform designed to run on Google Cloud Run with zero-cost hosting for low-traffic sites. Perfect for photographers and businesses who want to deploy multiple branded photo galleries with different configurations.

## Features

- 🖼️ **Beautiful Photo Galleries** - Responsive, modern UI with justified grid layout
- 🔐 **Private Albums** - Secure access with unique codes
- 🔞 **Optional Age Verification** - Configurable 18+ gate
- 🎨 **Customizable Branding** - Configure colors, logos, and site names
- ☁️ **Serverless** - Runs on Google Cloud Run (free tier eligible)
- 💾 **JSON Database** - No database costs, data stored in Cloud Storage
- 📸 **Automatic Thumbnails** - Smart image resizing and optimization
- 🚀 **Multi-Site Support** - Deploy multiple instances with different configs

## Quick Start

### Prerequisites

- Node.js 18+
- Google Cloud account
- `gcloud` CLI installed

### Local Development

1. Clone the repository:
   ```bash
   git clone https://github.com/YOUR_USERNAME/photowebsite.git
   cd photowebsite
   ```

2. Install dependencies:
   ```bash
   npm install
   ```

3. Copy `.env.example` to `.env` and configure:
   ```bash
   # Local development can use SQLite
   USE_JSON=false
   USE_CLOUD_STORAGE=false
   ADMIN_PASSWORD=changeme
   
   # Site configuration
   AGE_VERIFICATION_ENABLED=true
   PRIMARY_COLOR=#a6ff00
   LOGO_URL=/logo.png
   SITE_NAME=My Photo Gallery
   ```

4. Build CSS and run:
   ```bash
   npm run build:css
   npm start
   ```

5. Visit `http://localhost:3000`

## Deployment

See [DEPLOYMENT.md](./DEPLOYMENT.md) for full multi-site deployment instructions.

### Quick Deploy

```bash
# Set your project ID
PROJECT_ID=your-project-id

# Deploy to Cloud Run
gcloud run deploy photowebsite \
  --source . \
  --platform managed \
  --region europe-west1 \
  --allow-unauthenticated \
  --set-env-vars="USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${PROJECT_ID}-photos" \
  --set-env-vars="AGE_VERIFICATION_ENABLED=true,PRIMARY_COLOR=#a6ff00,LOGO_URL=/logo.png,SITE_NAME=My Gallery" \
  --set-secrets="ADMIN_PASSWORD=ADMIN_PASSWORD:latest" \
  --project=${PROJECT_ID}
```

## Configuration

All site-specific settings are controlled via environment variables:

| Variable | Description | Default | Example |
|----------|-------------|---------|---------|
| `AGE_VERIFICATION_ENABLED` | Enable 18+ age gate | `true` | `true` or `false` |
| `PRIMARY_COLOR` | Brand color (hex) | `#a6ff00` | `#ff6b00` |
| `LOGO_URL` | Logo path or URL | `/logo.png` | `https://...` |
| `SITE_NAME` | Site name/branding | `Lucky Pup Photography` | `Your Studio` |
| `ADMIN_PASSWORD` | Admin password | `changeme` | (secret) |
| `USE_JSON` | Use JSON database | `true` | `true` or `false` |
| `USE_CLOUD_STORAGE` | Use GCS for images | `true` | `true` or `false` |
| `GCS_BUCKET_NAME` | GCS bucket | (required) | `my-photos` |

## Multi-Site Deployment

This platform supports deploying multiple photo galleries from the same codebase:

```bash
# Site 1 - Wedding Photography
gcloud run deploy photowebsite-weddings \
  --source . \
  --set-env-vars="SITE_NAME=Wedding Photos,PRIMARY_COLOR=#ff99cc" \
  --region europe-west1 \
  --project=your-project

# Site 2 - Portrait Studio
gcloud run deploy photowebsite-portraits \
  --source . \
  --set-env-vars="SITE_NAME=Portrait Studio,PRIMARY_COLOR=#3366ff,AGE_VERIFICATION_ENABLED=false" \
  --region europe-west1 \
  --project=your-project
```

Each site gets its own:
- Cloud Run service
- Storage bucket
- Database (JSON file)
- Custom domain (optional)

## Architecture

- **Frontend**: Server-rendered EJS templates with vanilla JavaScript
- **Backend**: Express.js on Node.js 18
- **Database**: JSON file stored in Google Cloud Storage (or SQLite for local dev)
- **Storage**: Google Cloud Storage for photos and thumbnails
- **Hosting**: Google Cloud Run (serverless, auto-scaling)
- **Build**: Docker containerization

## Cost

Designed to run within Google Cloud's free tier:

- **Cloud Run**: 2M requests/month, 360K vCPU-seconds, 180K GiB-seconds free
- **Cloud Storage**: 5GB free storage, 5K Class A operations, 50K Class B operations
- **Artifact Registry**: 0.5GB free

For a low-traffic photo gallery (<10K visits/month), you can run this **completely free**.

## Admin Panel

Access the admin panel at `/admin` with HTTP Basic Auth:
- **Username**: `admin`
- **Password**: Your `ADMIN_PASSWORD` env var

Features:
- Create/edit/delete albums
- Upload photos (batch upload supported)
- Organize into groups
- Set public visibility
- Drag-and-drop reordering
- Generate access codes

## Custom Domain

To use a custom domain:

1. Deploy your service
2. Map domain in Cloud Run:
   ```bash
   gcloud run domain-mappings create \
     --service photowebsite \
     --domain yourdomain.com \
     --region europe-west1
   ```
3. Add DNS records (Cloud Run will provide them)
4. Wait for SSL certificate provisioning (5-15 minutes)

## Development

```bash
# Install dependencies
npm install

# Build CSS from SCSS
npm run build:css

# Start dev server (with auto-reload)
npm run dev

# Run linter
npm run lint
```

## License

MIT License - see LICENSE file

## Support

For issues or questions, open an issue on GitHub or contact the maintainer.

---

Built with ❤️ for photographers who want simple, beautiful photo galleries.
