<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\AI;

use Joomla\CMS\Http\HttpFactory;

defined('_JEXEC') or die;

/**
 * Anthropic (Claude) API Provider
 *
 * @since  1.0.0
 */
class AnthropicProvider implements AIProviderInterface
{
    /**
     * API Key
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiKey;

    /**
     * Model name
     *
     * @var    string
     * @since  1.0.0
     */
    private string $model;

    /**
     * API endpoint
     *
     * @var    string
     * @since  1.0.0
     */
    private string $endpoint = 'https://api.anthropic.com/v1/messages';

    /**
     * API version
     *
     * @var    string
     * @since  1.0.0
     */
    private string $apiVersion = '2023-06-01';

    /**
     * Constructor
     *
     * @param   string  $apiKey  Anthropic API key
     * @param   string  $model   Model name
     *
     * @since   1.0.0
     */
    public function __construct(string $apiKey, string $model = 'claude-3-5-sonnet-20241022')
    {
        $this->apiKey = $apiKey;
        $this->model = $model;
    }

    /**
     * Send a chat message and get response
     *
     * @param   array  $messages  Conversation history
     *
     * @return  string  AI response
     * @throws  \RuntimeException
     * @since   1.0.0
     */
    public function chat(array $messages): string
    {
        // Validate API key is set
        if (empty($this->apiKey)) {
            throw new \RuntimeException('Anthropic API key is not configured');
        }

        $http = HttpFactory::getHttp();

        // Anthropic API format: separate system message from messages
        $systemMessage = '';
        $anthropicMessages = [];

        foreach ($messages as $message) {
            if ($message['role'] === 'system') {
                $systemMessage = $message['content'];
            } else {
                $anthropicMessages[] = [
                    'role' => $message['role'] === 'assistant' ? 'assistant' : 'user',
                    'content' => $message['content']
                ];
            }
        }

        $payload = [
            'model' => $this->model,
            'messages' => $anthropicMessages,
            'max_tokens' => 4000,
            'temperature' => 0.7
        ];

        if ($systemMessage) {
            $payload['system'] = $systemMessage;
        }

        $headers = [
            'Content-Type' => 'application/json',
            'x-api-key' => $this->apiKey,
            'anthropic-version' => $this->apiVersion
        ];

        try {
            $response = $http->post(
                $this->endpoint,
                json_encode($payload),
                $headers,
                30
            );

            if ($response->code !== 200) {
                // Don't expose API key or full response body
                \Joomla\CMS\Log\Log::add(
                    "Anthropic API error: {$response->code} - {$response->body}",
                    \Joomla\CMS\Log\Log::ERROR,
                    'aiassistant'
                );
                throw new \RuntimeException(
                    "Anthropic API error (HTTP {$response->code}). Check error logs for details."
                );
            }

            $data = json_decode($response->body, true);

            if (json_last_error() !== JSON_ERROR_NONE || !isset($data['content'][0]['text'])) {
                throw new \RuntimeException('Invalid response from Anthropic API');
            }

            return $data['content'][0]['text'];

        } catch (\RuntimeException $e) {
            throw $e;
        } catch (\Exception $e) {
            \Joomla\CMS\Log\Log::add(
                'Anthropic communication error: ' . $e->getMessage(),
                \Joomla\CMS\Log\Log::ERROR,
                'aiassistant'
            );
            throw new \RuntimeException(
                'Failed to communicate with Anthropic. Check error logs for details.'
            );
        }
    }

    /**
     * Get provider name
     *
     * @return  string
     * @since   1.0.0
     */
    public function getName(): string
    {
        return 'Anthropic Claude (' . $this->model . ')';
    }

    /**
     * Validate API credentials
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validateCredentials(): bool
    {
        try {
            $this->chat([
                ['role' => 'user', 'content' => 'test']
            ]);
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }
}

