<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * List Database Tables Action
 *
 * @since  1.0.0
 */
class ListTablesAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  List of tables
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        $tables = $db->getTableList();
        $prefix = $db->getPrefix();

        // Filter to show only Joomla tables (with prefix)
        $joomlaTables = array_filter($tables, function($table) use ($prefix) {
            return strpos($table, $prefix) === 0;
        });

        // Group tables by component/purpose
        $grouped = $this->groupTables($joomlaTables, $prefix);

        return [
            'success' => true,
            'prefix' => $prefix,
            'total_count' => count($joomlaTables),
            'tables' => array_values($joomlaTables),
            'grouped' => $grouped
        ];
    }

    /**
     * Group tables by component/purpose
     *
     * @param   array   $tables  List of tables
     * @param   string  $prefix  Database prefix
     *
     * @return  array  Grouped tables
     * @since   1.0.0
     */
    private function groupTables(array $tables, string $prefix): array
    {
        $grouped = [
            'core' => [],
            'components' => [],
            'extensions' => []
        ];

        $coreKeywords = [
            'content', 'menu', 'modules', 'users', 'categories', 'extensions',
            'assets', 'session', 'fields', 'tags', 'banners', 'contact',
            'newsfeeds', 'redirect', 'finder', 'workflow'
        ];

        foreach ($tables as $table) {
            $shortName = str_replace($prefix, '#__', $table);
            
            $isCore = false;
            foreach ($coreKeywords as $keyword) {
                if (strpos($shortName, $keyword) !== false) {
                    $grouped['core'][] = $shortName;
                    $isCore = true;
                    break;
                }
            }

            if (!$isCore) {
                // Try to identify component
                if (preg_match('/#__([a-z]+)_/', $shortName, $matches)) {
                    $component = $matches[1];
                    if (!isset($grouped['components'][$component])) {
                        $grouped['components'][$component] = [];
                    }
                    $grouped['components'][$component][] = $shortName;
                } else {
                    $grouped['extensions'][] = $shortName;
                }
            }
        }

        return $grouped;
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'List all database tables in the Joomla site. Shows tables grouped by core, components, and extensions. Use this to discover what data is available.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}

