<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions;

use Joomla\CMS\Factory;

defined('_JEXEC') or die;

/**
 * Action Registry
 * Manages available actions and their permissions
 *
 * @since  1.0.0
 */
class ActionRegistry
{
    /**
     * Registered actions
     *
     * @var    array
     * @since  1.0.0
     */
    private array $actions = [];

    /**
     * Plugin parameters
     *
     * @var    array
     * @since  1.0.0
     */
    private array $params;

    /**
     * Constructor
     *
     * @param   array  $params  Plugin parameters
     *
     * @since   1.0.0
     */
    public function __construct(array $params)
    {
        $this->params = $params;
        $this->registerDefaultActions();
    }

    /**
     * Register default actions
     *
     * @return  void
     * @since   1.0.0
     */
    private function registerDefaultActions(): void
    {
        // Read actions
        $this->register('read_article', Actions\ReadArticleAction::class);
        $this->register('search_articles', Actions\SearchArticlesAction::class);
        $this->register('read_module', Actions\ReadModuleAction::class);
        $this->register('list_modules', Actions\ListModulesAction::class);
        $this->register('read_menu', Actions\ReadMenuAction::class);
        $this->register('read_pagebuilder_page', Actions\ReadPageBuilderPageAction::class);
        $this->register('get_site_info', Actions\GetSiteInfoAction::class);

        // Write actions
        $this->register('update_article', Actions\UpdateArticleAction::class);
        $this->register('create_article', Actions\CreateArticleAction::class);
        $this->register('update_module', Actions\UpdateModuleAction::class);
        $this->register('update_pagebuilder_page', Actions\UpdatePageBuilderPageAction::class);
        
        // Undo action
        $this->register('undo_action', Actions\UndoActionAction::class);
    }

    /**
     * Register an action
     *
     * @param   string  $name   Action name
     * @param   string  $class  Action class
     *
     * @return  void
     * @since   1.0.0
     */
    public function register(string $name, string $class): void
    {
        if (!class_exists($class)) {
            return;
        }

        if (!in_array(ActionInterface::class, class_implements($class))) {
            return;
        }

        $this->actions[$name] = $class;
    }

    /**
     * Get an action instance
     *
     * @param   string  $name  Action name
     *
     * @return  ActionInterface|null
     * @since   1.0.0
     */
    public function getAction(string $name): ?ActionInterface
    {
        if (!isset($this->actions[$name])) {
            return null;
        }

        if (!$this->isActionAllowed($name)) {
            throw new \RuntimeException("Action '{$name}' is not permitted");
        }

        $class = $this->actions[$name];
        return new $class();
    }

    /**
     * Get all available actions
     *
     * @return  array
     * @since   1.0.0
     */
    public function getAvailableActions(): array
    {
        $available = [];

        foreach ($this->actions as $name => $class) {
            if ($this->isActionAllowed($name)) {
                $available[$name] = $class;
            }
        }

        return $available;
    }

    /**
     * Check if action is allowed by configuration
     *
     * @param   string  $name  Action name
     *
     * @return  boolean
     * @since   1.0.0
     */
    private function isActionAllowed(string $name): bool
    {
        $allowedActions = $this->params['allowed_actions'] ?? [];

        // Map action names to permission keys
        $permissionMap = [
            'read_article' => 'read_articles',
            'search_articles' => 'read_articles',
            'update_article' => 'write_articles',
            'create_article' => 'write_articles',
            'read_module' => 'read_modules',
            'list_modules' => 'read_modules',
            'update_module' => 'write_modules',
            'read_menu' => 'read_menus',
            'read_pagebuilder_page' => 'read_pagebuilder',
            'update_pagebuilder_page' => 'write_pagebuilder',
            'get_site_info' => 'read_config',
            'undo_action' => 'write_articles', // Requires write permission to undo
        ];

        $requiredPermission = $permissionMap[$name] ?? null;

        if (!$requiredPermission) {
            return false;
        }

        return in_array($requiredPermission, $allowedActions);
    }
}

