<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions;

use Joomla\CMS\Factory;

defined('_JEXEC') or die;

/**
 * Action Registry
 * Manages available actions and their permissions
 *
 * @since  1.0.0
 */
class ActionRegistry
{
    /**
     * Registered actions
     *
     * @var    array
     * @since  1.0.0
     */
    private array $actions = [];

    /**
     * Plugin parameters
     *
     * @var    array
     * @since  1.0.0
     */
    private array $params;

    /**
     * Constructor
     *
     * @param   array  $params  Plugin parameters
     *
     * @since   1.0.0
     */
    public function __construct(array $params)
    {
        $this->params = $params;
        $this->registerDefaultActions();
    }

    /**
     * Register default actions
     *
     * @return  void
     * @since   1.0.0
     */
    private function registerDefaultActions(): void
    {
        // === UNIVERSAL DISCOVERY ACTIONS ===
        // These work on ANY Joomla site, regardless of installed components
        $this->register('query_database', Actions\QueryDatabaseAction::class);
        $this->register('list_tables', Actions\ListTablesAction::class);
        $this->register('describe_table', Actions\DescribeTableAction::class);
        $this->register('list_extensions', Actions\ListExtensionsAction::class);
        $this->register('get_site_info', Actions\GetSiteInfoAction::class);

        // === CORE JOOMLA WRITE ACTIONS ===
        // These use Joomla's API for safe, validated writes to core tables
        $this->register('update_article', Actions\UpdateArticleAction::class);
        $this->register('create_article', Actions\CreateArticleAction::class);
        $this->register('update_module', Actions\UpdateModuleAction::class);
        
        // === UNDO ===
        $this->register('undo_action', Actions\UndoActionAction::class);
        
        // NOTE: Component-specific read actions (like read_pagebuilder_page) are removed.
        // The AI should discover installed components using list_extensions and list_tables,
        // then use query_database to fetch data from ANY component's tables.
        // This makes the plugin universal and works with ANY Joomla site.
    }

    /**
     * Register an action
     *
     * @param   string  $name   Action name
     * @param   string  $class  Action class
     *
     * @return  void
     * @since   1.0.0
     */
    public function register(string $name, string $class): void
    {
        if (!class_exists($class)) {
            return;
        }

        if (!in_array(ActionInterface::class, class_implements($class))) {
            return;
        }

        $this->actions[$name] = $class;
    }

    /**
     * Get an action instance
     *
     * @param   string  $name  Action name
     *
     * @return  ActionInterface|null
     * @since   1.0.0
     */
    public function getAction(string $name): ?ActionInterface
    {
        if (!isset($this->actions[$name])) {
            return null;
        }

        if (!$this->isActionAllowed($name)) {
            // Get the permission required
            $permissionMap = [
                // Universal discovery actions
                'query_database' => 'query_database',
                'list_tables' => 'query_database',
                'describe_table' => 'query_database',
                'list_extensions' => 'query_database',
                'get_site_info' => 'read_config',
                // Core write actions
                'update_article' => 'write_articles',
                'create_article' => 'write_articles',
                'update_module' => 'write_modules',
                'undo_action' => 'write_articles',
            ];
            
            $requiredPermission = $permissionMap[$name] ?? 'unknown';
            
            throw new \RuntimeException(
                "Action '{$name}' is not permitted. Please enable '{$requiredPermission}' in plugin settings: " .
                "System → Plugins → AI Assistant → Basic Options → Allowed Actions"
            );
        }

        $class = $this->actions[$name];
        return new $class();
    }

    /**
     * Get all available actions
     *
     * @return  array
     * @since   1.0.0
     */
    public function getAvailableActions(): array
    {
        $available = [];

        foreach ($this->actions as $name => $class) {
            if ($this->isActionAllowed($name)) {
                $available[$name] = $class;
            }
        }

        return $available;
    }

    /**
     * Check if action is allowed by configuration
     *
     * @param   string  $name  Action name
     *
     * @return  boolean
     * @since   1.0.0
     */
    private function isActionAllowed(string $name): bool
    {
        $allowedActions = $this->params['allowed_actions'] ?? [];
        
        // If no actions are configured at all, default to allowing all read-only actions
        // This provides a better out-of-box experience
        if (empty($allowedActions)) {
            \Joomla\CMS\Log\Log::add(
                '[WARN] No allowed actions configured, defaulting to read-only actions. Configure in plugin settings for security.',
                \Joomla\CMS\Log\Log::WARNING,
                'aiassistant'
            );
            
            // Default to safe read-only actions INCLUDING dynamic query capabilities
            $allowedActions = [
                'query_database',    // NEW: Allow AI to query any table
                'read_articles', 
                'read_modules', 
                'read_menus', 
                'read_pagebuilder', 
                'read_config'
            ];
        }

        // Map action names to permission keys
        $permissionMap = [
            // Universal discovery actions
            'query_database' => 'query_database',
            'list_tables' => 'query_database',
            'describe_table' => 'query_database',
            'list_extensions' => 'query_database',
            'get_site_info' => 'read_config',
            // Core write actions
            'update_article' => 'write_articles',
            'create_article' => 'write_articles',
            'update_module' => 'write_modules',
            'undo_action' => 'write_articles', // Requires write permission to undo
        ];

        $requiredPermission = $permissionMap[$name] ?? null;

        if (!$requiredPermission) {
            return false;
        }

        return in_array($requiredPermission, $allowedActions);
    }
}

