<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Extension;

use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * Streaming Response Helper
 * Sends real-time updates to the frontend as the agent works
 *
 * @since  1.0.0
 */
class StreamingResponse
{
    /**
     * Whether streaming has started
     *
     * @var    boolean
     * @since  1.0.0
     */
    private static $started = false;

    /**
     * Start streaming response
     *
     * @return  void
     * @since   1.0.0
     */
    public static function start(): void
    {
        if (self::$started) {
            return;
        }

        // Log that streaming is starting
        Log::add('[STREAM] Starting SSE streaming', Log::DEBUG, 'aiassistant');

        // Clear all output buffers first
        while (ob_get_level() > 0) {
            ob_end_clean();
        }
        
        // Set headers for Server-Sent Events
        header('Content-Type: text/event-stream');
        header('Cache-Control: no-cache');
        header('Connection: keep-alive');
        header('X-Accel-Buffering: no'); // Disable nginx buffering
        
        self::$started = true;
        
        // Send an initial "alive" message to confirm stream is working
        echo "event: stream_started\n";
        echo "data: " . json_encode(['status' => 'connected', 'timestamp' => time()]) . "\n\n";
        flush();
        
        Log::add('[STREAM] Initial stream_started event sent', Log::DEBUG, 'aiassistant');
    }

    /**
     * Send an event to the client
     *
     * @param   string  $type  Event type
     * @param   mixed   $data  Event data
     *
     * @return  void
     * @since   1.0.0
     */
    public static function send(string $type, $data): void
    {
        if (!self::$started) {
            self::start();
        }

        $eventData = [
            'type' => $type,
            'data' => $data,
            'timestamp' => time()
        ];

        echo "event: {$type}\n";
        echo "data: " . json_encode($eventData) . "\n\n";
        
        // Flush output immediately
        if (ob_get_level()) {
            ob_flush();
        }
        flush();
    }

    /**
     * Send iteration start event
     *
     * @param   int     $iteration  Iteration number
     * @param   string  $thought    AI's thought for this iteration
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendIterationStart(int $iteration, string $thought): void
    {
        self::send('iteration_start', [
            'iteration' => $iteration,
            'thought' => $thought
        ]);
    }

    /**
     * Send iteration update event (to replace placeholder with real thought)
     *
     * @param   int     $iteration  Iteration number
     * @param   string  $thought    Updated AI thought
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendIterationUpdate(int $iteration, string $thought): void
    {
        self::send('iteration_update', [
            'iteration' => $iteration,
            'thought' => $thought
        ]);
    }

    /**
     * Send action execution event
     *
     * @param   string  $actionType  Type of action
     * @param   array   $parameters  Action parameters
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendActionExecuting(string $actionType, array $parameters): void
    {
        self::send('action_executing', [
            'type' => $actionType,
            'parameters' => $parameters
        ]);
    }

    /**
     * Send action result event
     *
     * @param   array  $action  Completed action with result
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendActionResult(array $action): void
    {
        self::send('action_result', [
            'type' => $action['type'],
            'status' => $action['status'],
            'result' => $action['result'] ?? null,
            'error' => $action['error'] ?? null
        ]);
    }

    /**
     * Send final completion event
     *
     * @param   array  $result  Final result
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendComplete(array $result): void
    {
        self::send('complete', $result);
    }

    /**
     * Send error event
     *
     * @param   string  $message  Error message
     *
     * @return  void
     * @since   1.0.0
     */
    public static function sendError(string $message): void
    {
        self::send('error', ['message' => $message]);
    }
}

