<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Agent;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Uri\Uri;
use Joomla\Plugin\System\AiAssistant\AI\AIProviderInterface;
use Joomla\Plugin\System\AiAssistant\Actions\ActionRegistry;
use Joomla\Plugin\System\AiAssistant\Logger\ActionLogger;
use Joomla\Plugin\System\AiAssistant\Extension\StreamingResponse;

defined('_JEXEC') or die;

/**
 * Agent Orchestrator - The brain of the AI Assistant
 *
 * This class implements the agentic reasoning loop:
 * 1. Receives user prompt
 * 2. Analyzes and plans actions
 * 3. Executes actions in sequence
 * 4. Verifies results
 * 5. Continues until goal is achieved or max iterations reached
 *
 * @since  1.0.0
 */
class AgentOrchestrator
{
    /**
     * AI Provider instance
     *
     * @var    AIProviderInterface
     * @since  1.0.0
     */
    private AIProviderInterface $aiProvider;

    /**
     * Action Registry instance
     *
     * @var    ActionRegistry
     * @since  1.0.0
     */
    private ActionRegistry $actionRegistry;

    /**
     * Action Logger instance
     *
     * @var    ActionLogger
     * @since  1.0.0
     */
    private ActionLogger $logger;

    /**
     * Plugin parameters
     *
     * @var    array
     * @since  1.0.0
     */
    private array $params;

    /**
     * Conversation history for context
     *
     * @var    array
     * @since  1.0.0
     */
    private array $conversationHistory = [];

    /**
     * Maximum iterations for agent loop
     *
     * @var    int
     * @since  1.0.0
     */
    private int $maxIterations = 20;

    /**
     * Maximum conversation history size
     *
     * @var    int
     * @since  1.0.0
     */
    private int $maxHistorySize = 50;

    /**
     * Constructor
     *
     * @param   AIProviderInterface  $aiProvider      AI provider instance
     * @param   ActionRegistry       $actionRegistry  Action registry instance
     * @param   ActionLogger         $logger          Logger instance
     * @param   array                $params          Plugin parameters
     *
     * @since   1.0.0
     */
    public function __construct(
        AIProviderInterface $aiProvider,
        ActionRegistry $actionRegistry,
        ActionLogger $logger,
        array $params
    ) {
        $this->aiProvider = $aiProvider;
        $this->actionRegistry = $actionRegistry;
        $this->logger = $logger;
        $this->params = $params;
        $this->maxIterations = (int) ($params['max_actions_per_request'] ?? 10);
    }

    /**
     * Process a user prompt through the agentic reasoning loop
     *
     * @param   string   $userPrompt    User's natural language request
     * @param   boolean  $requireReview Whether to require review before execution
     * @param   boolean  $streaming     Whether to stream responses in real-time
     *
     * @return  array    Result containing actions taken, summary, and status
     * @since   1.0.0
     */
    public function processPrompt(string $userPrompt, bool $requireReview = true, bool $streaming = false): array
    {
        $sessionId = $this->logger->startSession($userPrompt);
        
        try {
            // Initialize the conversation with system prompt and user request
            $this->initializeConversation($userPrompt);

            $iteration = 0;
            $allActions = [];
            $isComplete = false;
            $executedActionSignatures = []; // Track what we've already done to prevent duplicates
            $executedActionResults = []; // Store results by signature for reuse
            $iterationThoughts = []; // Capture AI's thinking at each iteration for display
            $askedForFinalAnswer = false; // Track if we've explicitly requested a final answer

            while ($iteration < $this->maxIterations && !$isComplete) {
                $iteration++;

                Log::add('[DEBUG] Starting iteration ' . $iteration, Log::DEBUG, 'aiassistant');

                // Get AI's next thought/action
                $response = $this->aiProvider->chat($this->conversationHistory);
                
                Log::add('[DEBUG] AI Response (first 500 chars): ' . substr($response, 0, 500), Log::DEBUG, 'aiassistant');
                
                // Add assistant response to history
                $this->conversationHistory[] = [
                    'role' => 'assistant',
                    'content' => $response
                ];

                // Parse the response to extract actions and thought
                $parsedActions = $this->parseAgentResponse($response);
                
                // Extract the "thought" from the parsed response for display
                $currentThought = $this->extractThought($response);
                if ($currentThought) {
                    $thoughtData = [
                        'iteration' => $iteration,
                        'thought' => $currentThought,
                        'action_count' => count($parsedActions)
                    ];
                    $iterationThoughts[] = $thoughtData;
                    
                    // Send iteration with actual thought to display
                    if ($streaming) {
                        StreamingResponse::sendIterationStart($iteration, $currentThought);
                    }
                }
                
                Log::add('[DEBUG] Parsed ' . count($parsedActions) . ' actions from response', Log::DEBUG, 'aiassistant');

                if (empty($parsedActions)) {
                    // No actions found - check if AI provided a final answer or is being lazy
                    $hasCompleteMarker = (stripos($response, 'COMPLETE') !== false);
                    $hasSubstantiveContent = (strlen(trim($response)) > 200); // More than just a short message
                    
                    if ($hasCompleteMarker && $hasSubstantiveContent) {
                        // AI has provided final answer
                        Log::add('[DEBUG] No actions found, AI provided final answer', Log::DEBUG, 'aiassistant');
                        $isComplete = true;
                        break;
                    } else {
                        // AI is being lazy or wrote an essay without actions
                        Log::add('[WARN] AI response has no actions and no complete answer - forcing to act', Log::WARNING, 'aiassistant');
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "❌ ERROR: You must either:\n1. Output JSON actions to get more data\n2. OR provide your FINAL ANSWER and say COMPLETE\n\nDo NOT write essays or plans. Just act!"
                        ];
                        continue;
                    }
                }

                // LIMIT to maximum 1 action per iteration to force true back-and-forth
                $maxActionsPerIteration = 1;
                if (count($parsedActions) > $maxActionsPerIteration) {
                    Log::add('[LIMIT] AI requested ' . count($parsedActions) . ' actions, limiting to ' . $maxActionsPerIteration . ' for back-and-forth flow', Log::INFO, 'aiassistant');
                    $parsedActions = array_slice($parsedActions, 0, $maxActionsPerIteration);
                }
                
                // Classify actions as read-only or write (and deduplicate)
                $readActions = [];
                $writeActions = [];
                
                foreach ($parsedActions as $action) {
                    // Create a signature for deduplication
                    $signature = $action['type'] . ':' . json_encode($action['parameters']);
                    
                    // Check if we've already executed this exact action
                    if (isset($executedActionResults[$signature])) {
                        Log::add('[DEBUG] Reusing result from duplicate action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        // Reuse the previous result and add to readActions so it gets fed back to AI
                        $cachedAction = $executedActionResults[$signature];
                        $readActions[] = $cachedAction;
                        continue;
                    }
                    
                    if ($this->isReadOnlyAction($action['type'])) {
                        $readActions[] = $action;
                        $executedActionSignatures[] = $signature;
                    } else {
                        $writeActions[] = $action;
                        $executedActionSignatures[] = $signature;
                    }
                }
                
                // Execute read-only actions immediately (gathering information)
                foreach ($readActions as $index => $action) {
                    // Skip if this is a cached result (it's already executed)
                    if (isset($action['result'])) {
                        Log::add('[DEBUG] Skipping execution of cached action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        continue;
                    }
                    
                    Log::add('[DEBUG] Auto-executing read action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                    
                    // Stream action start in real-time
                    if ($streaming) {
                        StreamingResponse::sendActionExecuting($action['type'], $action['parameters']);
                    }
                    
                    $action = $this->executeAction($action);
                    $actionId = $this->logger->logAction($sessionId, $action);
                    $action['id'] = $actionId;
                    $allActions[] = $action;
                    
                    // Cache this result for deduplication
                    $signature = $action['type'] . ':' . json_encode($action['parameters'] ?? []);
                    $executedActionResults[$signature] = $action;
                    
                    // Stream action result in real-time
                    if ($streaming) {
                        StreamingResponse::sendActionResult($action);
                    }
                    
                    // Smart follow-up: If we just queried #__menu and found SP Page Builder, auto-fetch that page
                    if ($action['type'] === 'query_database' && isset($action['result']['results'])) {
                        $followUpActions = $this->analyzeQueryForFollowUp($action['result']['results']);
                        
                        foreach ($followUpActions as $followUp) {
                            Log::add('[DEBUG] Auto-executing follow-up: ' . $followUp['type'], Log::DEBUG, 'aiassistant');
                            $followUp = $this->executeAction($followUp);
                            $followUpId = $this->logger->logAction($sessionId, $followUp);
                            $followUp['id'] = $followUpId;
                            $allActions[] = $followUp;
                            $readActions[] = $followUp;
                        }
                    }
                }
                
                // Handle write actions based on review setting
                foreach ($writeActions as $action) {
                    if ($requireReview) {
                        Log::add('[DEBUG] Queueing write action for review: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        $action['status'] = 'pending_review';
                        $this->logger->logAction($sessionId, $action);
                    } else {
                        Log::add('[DEBUG] Auto-executing write action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        $action = $this->executeAction($action);
                        $actionId = $this->logger->logAction($sessionId, $action);
                        $action['id'] = $actionId;
                    }
                    $allActions[] = $action;
                }

                // Feed execution results back to AI for read actions
                if (!empty($readActions)) {
                    $executionSummary = $this->generateExecutionSummary($readActions);
                    
                    // Check if any actions failed
                    $failedActions = array_filter($readActions, function($a) {
                        return isset($a['status']) && $a['status'] === 'failed';
                    });
                    
                    // Only force answer if we're approaching max iterations (safety net)
                    if ($iteration >= ($this->maxIterations - 2) && !$askedForFinalAnswer) {
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== EXECUTION RESULTS =====\n\n" . $executionSummary . "\n\n===== APPROACHING ITERATION LIMIT =====\nYou have " . ($this->maxIterations - $iteration) . " iterations remaining.\n\n📝 PROVIDE YOUR FINAL ANSWER NOW with the data you've collected.\n\nUser's question: \"{$userPrompt}\"\n\n⚠️ CRITICAL: Use the EXACT DATA from the results above. DO NOT make up generic content!\n⚠️ If you queried database tables, show the ACTUAL rows/data returned.\n⚠️ Be specific - use real titles, IDs, and values from the results.\n\n✅ Synthesize the SPECIFIC data you collected\n✅ Write a complete, detailed answer using REAL data\n✅ Do NOT hallucinate or guess - use what you actually retrieved!\n\n→ Write your answer using the exact data, THEN say COMPLETE"
                        ];
                        $askedForFinalAnswer = true;
                    } elseif (!empty($failedActions)) {
                        // If actions failed, explicitly guide retry
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== RESULTS =====\n\n" . $executionSummary . "\n\n===== ⚠️ FAILED ACTIONS DETECTED =====\n\n" .
                                        "Some of your queries FAILED. You must analyze the error messages and RETRY with corrections.\n\n" .
                                        "Common fixes:\n" .
                                        "- SQL syntax error? Check for MySQL reserved words like `fulltext`, `key`, `order` - use backticks!\n" .
                                        "- Table doesn't exist? Use list_tables to discover the correct table name\n" .
                                        "- Column doesn't exist? Use describe_table to see available columns\n\n" .
                                        "REQUEST 1 CORRECTED ACTION NOW (JSON format only)"
                        ];
                    } else {
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== RESULTS =====\n\n" . $executionSummary . "\n\n" .
                                        "===== INFORMATION SUFFICIENCY CHECK =====\n\n" .
                                        "User's question: \"{$userPrompt}\"\n\n" .
                                        "Before you can answer, verify you have ALL necessary information:\n\n" .
                                        "1. Do you have the specific data needed to answer this question?\n" .
                                        "2. If the question asks for \"all\" or \"each\" - do you have ALL items, not just some?\n" .
                                        "3. If the question asks for \"descriptions\" - do you have actual content, not just titles/metadata?\n" .
                                        "4. Are there any failed queries that prevented you from getting critical data?\n\n" .
                                        "===== YOUR RESPONSE =====\n\n" .
                                        ">> IF YOU HAVE ALL INFORMATION:\n" .
                                        "   Write a COMPLETE, DETAILED answer using the EXACT data above\n" .
                                        "   Then say COMPLETE\n\n" .
                                        ">> IF YOU'RE MISSING INFORMATION:\n" .
                                        "   Request 1 MORE action to gather the missing data (JSON format only)\n\n" .
                                        "RULES:\n" .
                                        "- EXACTLY 1 action per turn\n" .
                                        "- Use REAL data from results, don't make up content!\n" .
                                        "- Take your time - you have up to " . $this->maxIterations . " iterations total"
                        ];
                    }
                    
                    Log::add('[DEBUG] Fed results back to AI, continuing conversation (iteration ' . $iteration . '), askedForFinalAnswer=' . ($askedForFinalAnswer ? 'true' : 'false'), Log::DEBUG, 'aiassistant');
                    
                    // Trim conversation history if it gets too large
                    $this->trimConversationHistory();
                    
                    // Don't mark as complete yet - let AI process the results
                    continue;
                }
                
                // For write actions with review, mark as complete
                if ($requireReview && !empty($writeActions)) {
                    $isComplete = true;
                }

                // Check if agent says it's complete
                // But ONLY if we've actually done some work (gathered info or taken actions)
                if ((stripos($response, 'COMPLETE') !== false || stripos($response, 'task is finished') !== false) && !empty($allActions)) {
                    // Check if the response contains a proper answer (not just COMPLETE)
                    $responseWithoutJson = preg_replace('/```json.*?```/s', '', $response);
                    $responseWithoutJson = trim(str_replace('COMPLETE', '', $responseWithoutJson));
                    
                    // Quality checks for completeness
                    $answerLength = strlen($responseWithoutJson);
                    $endsIncomplete = preg_match('/\b(featuring|including|such as|like|with|for|and|or|but|the|a|an)\s*$/i', $responseWithoutJson);
                    
                    // For "list" or "describe" queries, expect substantial content
                    $needsDetailedAnswer = (stripos($userPrompt, 'list') !== false || 
                                          stripos($userPrompt, 'describe') !== false || 
                                          stripos($userPrompt, 'each') !== false ||
                                          stripos($userPrompt, 'all') !== false);
                    $minLength = $needsDetailedAnswer ? 200 : 100;
                    
                    if ($answerLength > $minLength && !$endsIncomplete) {
                        // Good - AI provided a complete answer
                        Log::add('[COMPLETE-CHECK] Answer is complete (' . $answerLength . ' chars)', Log::INFO, 'aiassistant');
                        $isComplete = true;
                    } else {
                        // AI's answer is too short or incomplete - force a better one
                        $reason = $endsIncomplete ? 'ends mid-sentence' : 'too short (' . $answerLength . ' < ' . $minLength . ' chars)';
                        Log::add('[WARN] AI said COMPLETE but answer is ' . $reason . ', forcing final answer', Log::WARNING, 'aiassistant');
                        
                        $executionSummary = $this->generateExecutionSummary($allActions);
                        
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "⚠️ Your answer is INCOMPLETE (reason: {$reason})!\n\n" .
                                        "User's question: \"{$userPrompt}\"\n\n" .
                                        "All the data you collected:\n\n{$executionSummary}\n\n" .
                                        "===== YOUR TASK =====\n" .
                                        "Provide a COMPLETE, DETAILED answer using ALL the data above.\n" .
                                        "- Use REAL data from the results (titles, IDs, actual content)\n" .
                                        "- Be specific and thorough\n" .
                                        "- Format it nicely for the user\n" .
                                        "- DON'T cut off mid-sentence!\n" .
                                        "- For 'list all X', include ALL items, not just one\n\n" .
                                        "Write your FULL answer NOW, then say COMPLETE:"
                        ];
                        continue;
                    }
                } elseif ((stripos($response, 'COMPLETE') !== false || stripos($response, 'task is finished') !== false) && empty($allActions)) {
                    // AI is trying to complete without doing anything - warn and continue
                    Log::add('[WARN] AI tried to complete without taking any actions, forcing continuation', Log::WARNING, 'aiassistant');
                    $this->conversationHistory[] = [
                        'role' => 'user',
                        'content' => "You haven't gathered any information yet! Please use the available actions (query_database, list_tables, describe_table) to gather the information needed to answer the user's question."
                    ];
                    continue;
                }
            }

            // Get the last AI response for informational queries
            $lastAiResponse = '';
            foreach (array_reverse($this->conversationHistory) as $msg) {
                if ($msg['role'] === 'assistant') {
                    $lastAiResponse = $msg['content'];
                    break;
                }
            }
            
            // Extract clean answer (remove JSON blocks if present)
            $cleanAnswer = $this->extractCleanAnswer($lastAiResponse);
            
            // CRITICAL: Check if we have failed actions that need retry
            $hasFailedActions = false;
            foreach ($allActions as $action) {
                if (isset($action['status']) && $action['status'] === 'failed') {
                    $hasFailedActions = true;
                    break;
                }
            }
            
            // If we have failed actions, don't force answer yet - AI should retry
            if ($hasFailedActions && !$isComplete) {
                Log::add('[FORCE-ANSWER] Skipping - there are failed actions that may need retry', Log::INFO, 'aiassistant');
            }
            // If we have actions but no substantive answer, force one final synthesis
            elseif (!empty($allActions) && strlen($cleanAnswer) < 100) {
                Log::add('[FORCE-ANSWER] AI completed actions but provided no answer. Forcing final synthesis.', Log::WARNING, 'aiassistant');
                
                // Build a summary of all the data collected
                $allResults = $this->generateExecutionSummary($allActions);
                
                // Count successful actions
                $successfulActions = array_filter($allActions, function($a) {
                    return isset($a['status']) && $a['status'] === 'executed';
                });
                
                $this->conversationHistory[] = [
                    'role' => 'user',
                    'content' => "===== FINAL SYNTHESIS REQUIRED =====\n\n" .
                                "You have completed " . count($successfulActions) . " successful actions (out of " . count($allActions) . " total).\n\n" .
                                "User's original question: \"{$userPrompt}\"\n\n" .
                                "All the data you collected:\n\n{$allResults}\n\n" .
                                "===== YOUR TASK =====\n" .
                                "Using the EXACT data above, provide a COMPLETE answer to the user's question.\n" .
                                "⚠️ If some queries failed and you're missing critical data, SAY SO - don't make up content!\n" .
                                "- Use REAL data from the successful results (IDs, titles, values)\n" .
                                "- Be specific and detailed\n" .
                                "- Format it nicely for the user\n" .
                                "- Do NOT request more actions\n\n" .
                                "Write your final answer NOW:"
                ];
                
                // Get final synthesis
                $finalResponse = $this->aiProvider->chat($this->conversationHistory);
                $this->conversationHistory[] = [
                    'role' => 'assistant',
                    'content' => $finalResponse
                ];
                
                $cleanAnswer = $this->extractCleanAnswer($finalResponse);
                Log::add('[FORCE-ANSWER] Received final answer (' . strlen($cleanAnswer) . ' chars)', Log::INFO, 'aiassistant');
            }
            
            $result = [
                'success' => true,
                'session_id' => $sessionId,
                'actions' => $allActions,
                'iterations' => $iteration,
                'iteration_thoughts' => $iterationThoughts,  // NEW: Show thinking at each step
                'status' => $requireReview ? 'pending_review' : 'executed',
                'summary' => $this->generateFinalSummary($allActions, $isComplete),
                'conversation' => $this->conversationHistory,
                'ai_response' => $cleanAnswer  // Clean AI response without JSON
            ];

            $this->logger->endSession($sessionId, $result);

            // Stream final completion
            if ($streaming) {
                StreamingResponse::sendComplete($result);
            }

            return $result;

        } catch (\Exception $e) {
            // Log detailed error internally
            Log::add(
                'AI Assistant session error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString(),
                Log::ERROR,
                'aiassistant'
            );

            $error = [
                'success' => false,
                'session_id' => $sessionId,
                'error' => $e->getMessage()
                // Removed trace from public response for security
            ];

            $this->logger->logError($sessionId, $error);

            return $error;
        }
    }

    /**
     * Execute previously planned actions after review
     *
     * @param   string  $sessionId  Session ID from planning phase
     * @param   array   $actionIds  Array of action IDs to execute
     *
     * @return  array   Execution results
     * @since   1.0.0
     */
    public function executeReviewedActions(string $sessionId, array $actionIds = []): array
    {
        $actions = $this->logger->getSessionActions($sessionId);
        $results = [];

        foreach ($actions as $action) {
            // If specific IDs provided, only execute those
            if (!empty($actionIds) && !in_array($action['id'], $actionIds)) {
                continue;
            }

            if ($action['status'] === 'pending_review') {
                $executedAction = $this->executeAction($action);
                $results[] = $executedAction;
                $this->logger->updateAction($action['id'], $executedAction);
            }
        }

        return [
            'success' => true,
            'executed_actions' => $results,
            'summary' => $this->generateExecutionSummary($results)
        ];
    }

    /**
     * Initialize conversation with system prompt and context
     *
     * @param   string  $userPrompt  User's request
     *
     * @return  void
     * @since   1.0.0
     */
    private function initializeConversation(string $userPrompt): void
    {
        $availableActions = $this->actionRegistry->getAvailableActions();
        $actionDescriptions = [];

        foreach ($availableActions as $actionName => $actionClass) {
            $actionDescriptions[] = "- {$actionName}: " . $actionClass::getDescription();
        }

        $systemPrompt = <<<SYSTEM
You are an AI assistant with FULL ACCESS to a Joomla 5 CMS. You can explore and query ANY data in the system.

🔍 **YOU ARE NOT LIMITED TO PREDEFINED ACTIONS!**

You have dynamic database access and can discover everything on your own:

{$this->formatActionList($availableActions)}

## 🚀 YOUR SUPERPOWERS:

### 1. **DISCOVER THE SITE** (Use these FIRST when encountering a new task):
- `list_tables` - See ALL database tables available
- `describe_table` - Understand table structure (columns, types, row counts)
- `list_extensions` - Find installed components/modules/plugins
- `query_database` - Execute ANY SELECT query on ANY table

### 2. **DYNAMIC QUERYING** (Your most powerful tool):
Use `query_database` with ANY SELECT statement:
```json
{
  "type": "query_database",
  "parameters": {
    "query": "SELECT * FROM #__content WHERE state=1 AND catid=9",
    "limit": 100
  },
  "reason": "Finding published articles in category 9"
}
```

Core Joomla tables (always present):
- `#__content` - Articles
- `#__modules` - Modules  
- `#__menu` - Menu items
- `#__categories` - Categories
- `#__extensions` - Installed extensions
- `#__template_styles` - Templates

**To discover component tables**: Use `list_tables` to see ALL available tables, then query them!

### 3. **UNIVERSAL PRINCIPLES FOR ANY QUESTION**:

**Think Before You Act:**
Before querying, ask yourself:
- What SPECIFIC data do I need to answer this question?
- Is metadata enough, or do I need full content?
- Are there multiple items that need individual lookups?
- What tables/components might contain this data?

**Discovery-First Approach:**
1. If you don't know where data is stored:
   - Use `list_tables` to see what's available
   - Use `describe_table` to understand structure
   - Use `list_extensions` to see what's installed

2. Don't assume components exist - verify first!

**Multi-Item Gathering:**
If a question involves multiple items (e.g., "list all X", "describe each Y"):
- Query to get the list of items
- For EACH item, query to get its detailed data
- Don't stop at just IDs/titles - get the actual content if needed for descriptions

**Keywords That Signal Depth Needed:**
- "describe", "description", "details" → Need FULL content, not just titles
- "each", "all", "list" → Need to iterate through multiple items
- "what's in/on" → Need to discover and fetch actual content
- "how many", "count" → Metadata might be enough

**SQL Query Best Practices:**
- ⚠️ **ALWAYS use backticks** for these common MySQL reserved words when used as column names:
  `fulltext`, `key`, `keys`, `order`, `group`, `rank`, `count`, `limit`, `table`, `index`, 
  `option`, `read`, `write`, `left`, `right`, `join`, `natural`, `default`, `check`, `desc`
- Example: SELECT `id`, `title`, `fulltext` FROM #__content (NOT: SELECT fulltext)
- If a query fails with "SQL syntax error", check error message for reserved words and retry with backticks
- Also check: correct table name, correct column names (use describe_table if unsure)

**Handle Failed Actions:**
- If an action fails, READ THE ERROR MESSAGE carefully
- For SQL errors: Check for reserved words, missing quotes, wrong table names
- RETRY with a corrected query - don't give up after one failure
- Don't provide a final answer if critical queries failed - gather the missing data first

**Information Sufficiency Check (THE CORE PRINCIPLE):**
🎯 **After EVERY action result, ask yourself:**
1. Do I have ALL the data needed to answer the user's question?
2. If they asked for "all/each" - do I have ALL items?
3. If they asked for "descriptions" - do I have actual content (not just titles)?
4. Did any queries fail that would prevent a complete answer?

✅ **HAVE ALL INFO?** → Provide complete, detailed answer + say COMPLETE
❌ **MISSING INFO?** → Request 1 more action to get missing data

**Never answer until you have complete information!**

**Always Synthesize:**
After gathering data, provide a complete, formatted answer.
Don't just list query results - interpret and present them clearly.

## 📋 **OPERATION WORKFLOW (TRUE BACK-AND-FORTH):**

**🚨 CRITICAL LIMIT**: You can request EXACTLY **1 ACTION** per turn!

Each iteration (strict back-and-forth):
1. Output 1 JSON action (EXACTLY ONE!)
2. System executes → returns results  
3. **CHECK: Do I have ALL info to answer now?**
   - YES → Write complete answer + say COMPLETE
   - NO → Continue to step 4
4. Decide what data is still missing
5. Output next 1 action to get missing data → repeat from step 2

**The system enforces a 1-action limit**. If you output multiple actions, only the first will execute!

**DO NOT** write essays! Output 1 JSON action → wait → get result → think → next action.

## ⚠️ **CRITICAL RULES:**

- **ONE AT A TIME**: Request EXACTLY 1 action per turn, get result, then decide next action
- **TRUE ITERATIVE**: You must see the result before deciding the next action
- **NO BATCH PLANNING**: Don't plan multiple actions ahead - see the result first!
- **SYNTHESIZE AT END**: Only write a detailed answer when you have ALL the data and say COMPLETE
- **TAKE YOUR TIME**: You have up to 20 iterations - use as many as needed to gather complete data!

## 🚫 **WHAT NOT TO DO:**

❌ "I plan to do X, then Y, then Z..." → NO! Just do X, wait for result.
❌ Outputting 2+ actions in the JSON → NO! System will truncate to 1.
❌ "I need to gather A, B, C, D..." → NO! Request A, see result, then decide.
❌ "Would you like me to proceed?" → NO! Just proceed immediately.
❌ **Making up generic/example data** → NO! Use ONLY the exact data you retrieved!
❌ **Hallucinating content** → NO! If you queried the database, use the ACTUAL rows returned!

## ✅ **WHAT TO DO:**

✅ Output 1 JSON action immediately
✅ Wait for result
✅ Analyze the specific result you got
✅ Based on that result, decide next action
✅ **Use EXACT data from query results** - real IDs, titles, values
✅ Provide final answer with REAL data when done

## 🛑 **ANTI-HALLUCINATION RULE:**

When you provide your final answer, you MUST use the EXACT data from the query results.
- If you queried ANY table, show the ACTUAL rows with real IDs, titles, and values returned
- If you queried `#__menu`, show the ACTUAL menu items returned
- If you queried `#__content`, show the ACTUAL article titles returned
- DO NOT make up "typical pages" like "About, Services, Contact" if you didn't see them in the data
- If the query returned empty results, SAY SO - don't invent content!
- If a component table doesn't exist, say "Component X not found" - don't pretend you queried it!

## 🎯 **ACTION FORMAT (CRITICAL):**

**Your "thought" field is shown to the user**, so make it clear and helpful!

```json
{
  "thought": "Need to find the homepage menu item to see what content it links to",
  "actions": [
    {"type": "query_database", "parameters": {"query": "SELECT * FROM #__menu WHERE home=1"}, "reason": "Get homepage menu"}
  ]
}
```

**Good thoughts:**
- ✅ "Found homepage menu item, now checking what component it links to"
- ✅ "Homepage links to com_example page 9, listing tables to find the page data"
- ✅ "Got the page data, now synthesizing the complete homepage content"

**Bad thoughts:**
- ❌ "Executing query" (too vague)
- ❌ "" (empty - always provide a thought!)
- ❌ "I plan to do X, Y, Z..." (don't plan, just do the next step)

Current site:
- Site: {$this->getSiteName()}
- Joomla: {JVERSION}
- URL: {$this->getSiteUrl()}

## 📊 **CORE JOOMLA TABLES (Always Present):**

**Content & Structure:**
- `#__content` - Articles (id, title, alias, introtext, fulltext, state, catid, created, modified, featured)
- `#__categories` - Categories (id, title, alias, description, parent_id, published)
- `#__menu` - Menu items (id, menutype, title, link, home, published, client_id)
- `#__modules` - Modules (id, title, module, position, published, params, client_id)

**Users & Access:**
- `#__users` - Users (id, name, username, email, registerDate, lastvisitDate, block)
- `#__usergroups` - User groups (id, title, parent_id)

**Extensions:**
- `#__extensions` - Installed extensions (extension_id, name, type, element, enabled)
- `#__template_styles` - Templates (id, template, title, home, params)

**Component Tables?** Every Joomla site is different!
- Use `list_extensions` to see what components are installed
- Use `list_tables` to see all available tables
- Look for patterns like `#__componentname_*` for component tables
- Use `describe_table` to understand table structure before querying

**YOU ARE A POWERFUL AGENT. EXPLORE. DISCOVER. ANSWER.**
SYSTEM;

        $this->conversationHistory = [
            ['role' => 'system', 'content' => $systemPrompt],
            ['role' => 'user', 'content' => $userPrompt]
        ];
    }

    /**
     * Parse agent response to extract structured actions
     *
     * @param   string  $response  AI response text
     *
     * @return  array   Array of parsed actions
     * @since   1.0.0
     */
    private function parseAgentResponse(string $response): array
    {
        $actions = [];
        $jsonData = null;

        // Method 1: Try to extract JSON from code blocks first
        if (preg_match('/```json\s*(\{(?:[^`]|`(?!``))*\})\s*```/s', $response, $matches)) {
            $jsonData = json_decode($matches[1], true);
            Log::add('[DEBUG] Found JSON in code block', Log::DEBUG, 'aiassistant');
        }
        
        // Method 2: Try to find raw JSON object in the response
        if (!$jsonData && preg_match('/\{[^{}]*"thought"[^{}]*"actions"[^{}]*\[[^\]]*\][^{}]*\}/s', $response, $matches)) {
            $jsonData = json_decode($matches[0], true);
            Log::add('[DEBUG] Found raw JSON object', Log::DEBUG, 'aiassistant');
        }
        
        // Method 3: More aggressive - find any valid JSON with "actions" key
        if (!$jsonData) {
            // Look for balanced braces that contain "actions"
            if (preg_match('/(\{(?:[^{}]|(?R))*\})/s', $response, $matches)) {
                $possibleJson = $matches[1];
                if (strpos($possibleJson, '"actions"') !== false) {
                    $jsonData = json_decode($possibleJson, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        Log::add('[DEBUG] Found JSON via aggressive search', Log::DEBUG, 'aiassistant');
                    } else {
                        $jsonData = null;
                    }
                }
            }
        }
        
        if ($jsonData && json_last_error() === JSON_ERROR_NONE && isset($jsonData['actions']) && is_array($jsonData['actions'])) {
            Log::add('[DEBUG] Successfully parsed JSON, found ' . count($jsonData['actions']) . ' actions', Log::DEBUG, 'aiassistant');
            
            foreach ($jsonData['actions'] as $action) {
                if (isset($action['type'])) {
                    $actions[] = [
                        'type' => $this->sanitizeString($action['type']),
                        'parameters' => $this->sanitizeParameters($action['parameters'] ?? []),
                        'reason' => $this->sanitizeString($action['reason'] ?? ''),
                        'thought' => $this->sanitizeString($jsonData['thought'] ?? ''),
                        'status' => 'planned'
                    ];
                }
            }
        } else {
            Log::add('[DEBUG] Could not parse actions from response. Response (first 500 chars): ' . substr($response, 0, 500), Log::DEBUG, 'aiassistant');
        }

        return $actions;
    }

    /**
     * Sanitize a string parameter
     *
     * @param   mixed  $value  Value to sanitize
     *
     * @return  string  Sanitized string
     * @since   1.0.0
     */
    private function sanitizeString($value): string
    {
        if (!is_string($value)) {
            return '';
        }
        // Remove any null bytes and trim
        return trim(str_replace("\0", '', $value));
    }

    /**
     * Sanitize parameters array
     *
     * @param   array  $parameters  Parameters to sanitize
     *
     * @return  array  Sanitized parameters
     * @since   1.0.0
     */
    private function sanitizeParameters(array $parameters): array
    {
        $sanitized = [];
        
        foreach ($parameters as $key => $value) {
            // Sanitize key
            $cleanKey = preg_replace('/[^a-zA-Z0-9_]/', '', $key);
            
            if (is_array($value)) {
                $sanitized[$cleanKey] = $this->sanitizeParameters($value);
            } elseif (is_string($value)) {
                $sanitized[$cleanKey] = $this->sanitizeString($value);
            } elseif (is_numeric($value) || is_bool($value)) {
                $sanitized[$cleanKey] = $value;
            }
        }
        
        return $sanitized;
    }

    /**
     * Execute a single action
     *
     * @param   array  $action  Action definition
     *
     * @return  array  Action with execution result
     * @since   1.0.0
     */
    private function executeAction(array $action): array
    {
        try {
            $actionHandler = $this->actionRegistry->getAction($action['type']);
            
            if (!$actionHandler) {
                throw new \RuntimeException("Unknown action type: {$action['type']}");
            }

            $result = $actionHandler->execute($action['parameters']);
            
            $action['status'] = 'executed';
            $action['result'] = $result;
            $action['executed_at'] = date('Y-m-d H:i:s');
            
            // Store before_state if available (for undo functionality)
            if (isset($result['before_state'])) {
                $action['before_state'] = $result['before_state'];
                unset($result['before_state']); // Don't show in result
                $action['result'] = $result;
            }

        } catch (\Exception $e) {
            $action['status'] = 'failed';
            $action['error'] = $e->getMessage();
        }

        return $action;
    }

    /**
     * Generate execution summary for feedback to AI
     *
     * @param   array  $actions  Array of actions
     *
     * @return  string  Summary text
     * @since   1.0.0
     */
    private function generateExecutionSummary(array $actions): string
    {
        $summary = [];

        foreach ($actions as $action) {
            $status = $action['status'] ?? 'unknown';
            $type = $action['type'];
            
            if ($status === 'executed') {
                // Format the result in a readable way for the AI
                $result = $action['result'];
                if (is_array($result)) {
                    // Pretty print for better AI understanding
                    $resultText = json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                } else {
                    $resultText = (string) $result;
                }
                $summary[] = "Action: {$type}\nStatus: Success\nResult:\n{$resultText}\n";
            } elseif ($status === 'failed') {
                $summary[] = "Action: {$type}\nStatus: Failed\nError: {$action['error']}\n";
            }
        }

        return implode("\n---\n", $summary);
    }

    /**
     * Generate final summary for user
     *
     * @param   array    $actions     All actions taken
     * @param   boolean  $isComplete  Whether task completed successfully
     *
     * @return  string   Human-readable summary
     * @since   1.0.0
     */
    private function generateFinalSummary(array $actions, bool $isComplete): string
    {
        $total = count($actions);
        $executed = count(array_filter($actions, fn($a) => $a['status'] === 'executed'));
        $failed = count(array_filter($actions, fn($a) => $a['status'] === 'failed'));
        $pending = count(array_filter($actions, fn($a) => $a['status'] === 'pending_review'));

        $summary = "Task " . ($isComplete ? "completed" : "processed") . ".\n\n";
        $summary .= "Actions: {$total} total";
        
        if ($executed > 0) {
            $summary .= ", {$executed} executed";
        }
        if ($failed > 0) {
            $summary .= ", {$failed} failed";
        }
        if ($pending > 0) {
            $summary .= ", {$pending} pending review";
        }

        return $summary;
    }

    /**
     * Format action list for system prompt
     *
     * @param   array  $actions  Available actions
     *
     * @return  string  Formatted list
     * @since   1.0.0
     */
    private function formatActionList(array $actions): string
    {
        $list = [];
        
        foreach ($actions as $name => $class) {
            $list[] = "- **{$name}**: " . $class::getDescription();
        }

        return implode("\n", $list);
    }

    /**
     * Get site name
     *
     * @return  string
     * @since   1.0.0
     */
    private function getSiteName(): string
    {
        $app = Factory::getApplication();
        return $app ? $app->get('sitename', 'Joomla Site') : 'Joomla Site';
    }

    /**
     * Get site URL
     *
     * @return  string
     * @since   1.0.0
     */
    private function getSiteUrl(): string
    {
        return Uri::root();
    }

    /**
     * Check if action is read-only (information gathering) vs write (modifying)
     *
     * @param   string  $actionType  Type of action
     *
     * @return  boolean  True if read-only
     * @since   1.0.0
     */
    private function isReadOnlyAction(string $actionType): bool
    {
        $readOnlyActions = [
            // Universal discovery actions (work on ANY Joomla site)
            'query_database',
            'list_tables',
            'describe_table',
            'list_extensions',
            'get_site_info'
        ];
        
        return in_array($actionType, $readOnlyActions);
    }

    /**
     * Trim conversation history to prevent memory issues
     *
     * @return  void
     * @since   1.0.0
     */
    private function trimConversationHistory(): void
    {
        if (count($this->conversationHistory) > $this->maxHistorySize) {
            // Keep system message and recent messages
            $systemMessage = $this->conversationHistory[0];
            $recentMessages = array_slice($this->conversationHistory, -($this->maxHistorySize - 1));
            $this->conversationHistory = array_merge([$systemMessage], $recentMessages);
        }
    }

    /**
     * Analyze query results and generate intelligent follow-up actions
     * 
     * DISABLED: Auto-follow-up was too prescriptive and assumed component knowledge.
     * The AI should discover and query tables on its own through iterative exploration.
     *
     * @param   array  $results  Query results
     *
     * @return  array  Follow-up actions to execute
     * @since   1.0.0
     */
    private function analyzeQueryForFollowUp(array $results): array
    {
        // DISABLED: Let the AI discover and query tables itself
        return [];
        
        $followUps = [];
        
        foreach ($results as $row) {
            // Convert stdClass to array if needed
            if (is_object($row)) {
                $row = json_decode(json_encode($row), true);
            }
            
            // Check if this is a menu item
            if (isset($row['link']) && isset($row['menutype'])) {
                $link = $row['link'];
                
                // SP Page Builder page
                if (strpos($link, 'com_sppagebuilder') !== false && strpos($link, 'view=page') !== false) {
                    if (preg_match('/id=(\d+)/', $link, $matches)) {
                        $pageId = (int) $matches[1];
                        Log::add('[SMART-FOLLOW-UP] Found SP Page Builder page ID: ' . $pageId, Log::INFO, 'aiassistant');
                        
                        // SP Page Builder stores pages in #__sppagebuilder (not #__sppagebuilder_pages)
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__sppagebuilder WHERE id=" . $pageId
                            ],
                            'reason' => 'Auto-fetching SP Page Builder page from #__sppagebuilder table',
                            'status' => 'planned'
                        ];
                        
                        // Get page addons (the actual content elements)
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__sppagebuilder_addons WHERE page_id=" . $pageId . " ORDER BY ordering"
                            ],
                            'reason' => 'Getting page content addons from #__sppagebuilder_addons',
                            'status' => 'planned'
                        ];
                    }
                }
                // Single article
                elseif (strpos($link, 'com_content') !== false && strpos($link, 'view=article') !== false) {
                    if (preg_match('/id=(\d+)/', $link, $matches)) {
                        $articleId = (int) $matches[1];
                        Log::add('[SMART-FOLLOW-UP] Found article ID: ' . $articleId, Log::INFO, 'aiassistant');
                        
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__content WHERE id=" . $articleId
                            ],
                            'reason' => 'Auto-fetching article content for article ID ' . $articleId,
                            'status' => 'planned'
                        ];
                    }
                }
                // Featured articles
                elseif (strpos($link, 'view=featured') !== false) {
                    Log::add('[SMART-FOLLOW-UP] Found featured articles layout', Log::INFO, 'aiassistant');
                    
                    $followUps[] = [
                        'type' => 'query_database',
                        'parameters' => [
                            'query' => "SELECT * FROM #__content WHERE featured=1 AND state=1 ORDER BY created DESC LIMIT 10"
                        ],
                        'reason' => 'Auto-fetching featured articles',
                        'status' => 'planned'
                    ];
                }
            }
        }
        
        return $followUps;
    }

    /**
     * Extract clean answer from AI response (remove JSON action blocks)
     *
     * @param   string  $response  Raw AI response
     *
     * @return  string  Clean answer text
     * @since   1.0.0
     */
    private function extractCleanAnswer(string $response): string
    {
        // Remove JSON code blocks
        $clean = preg_replace('/```json\s*\{[^`]*\}\s*```/s', '', $response);
        
        // Remove raw JSON objects (action plans)
        $clean = preg_replace('/\{[^{}]*"thought"[^{}]*"actions"[^{}]*\[[^\]]*\][^{}]*\}/s', '', $clean);
        
        // Remove "COMPLETE" marker
        $clean = str_ireplace('COMPLETE', '', $clean);
        
        // Clean up extra whitespace
        $clean = preg_replace('/\n{3,}/', "\n\n", $clean);
        $clean = trim($clean);
        
        // If nothing left, return a default message
        if (empty($clean)) {
            return "Actions completed. See details below.";
        }
        
        return $clean;
    }

    /**
     * Extract the "thought" field from AI's response
     *
     * @param   string  $response  AI response
     *
     * @return  string  Extracted thought or empty string
     * @since   1.0.0
     */
    private function extractThought(string $response): string
    {
        // Try to extract thought from JSON
        if (preg_match('/"thought"\s*:\s*"([^"]+)"/s', $response, $matches)) {
            return $matches[1];
        }
        
        // If no JSON thought, use the first sentence/line of the response
        $lines = explode("\n", trim($response));
        $firstLine = trim($lines[0] ?? '');
        
        // If it's not JSON and has substance, return it
        if ($firstLine && strpos($firstLine, '{') !== 0 && strlen($firstLine) > 10) {
            return substr($firstLine, 0, 200); // Limit length
        }
        
        return '';
    }

}

