<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Agent;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Uri\Uri;
use Joomla\Plugin\System\AiAssistant\AI\AIProviderInterface;
use Joomla\Plugin\System\AiAssistant\Actions\ActionRegistry;
use Joomla\Plugin\System\AiAssistant\Logger\ActionLogger;
use Joomla\Plugin\System\AiAssistant\Extension\StreamingResponse;

defined('_JEXEC') or die;

/**
 * Agent Orchestrator - The brain of the AI Assistant
 *
 * This class implements the agentic reasoning loop:
 * 1. Receives user prompt
 * 2. Analyzes and plans actions
 * 3. Executes actions in sequence
 * 4. Verifies results
 * 5. Continues until goal is achieved or max iterations reached
 *
 * @since  1.0.0
 */
class AgentOrchestrator
{
    /**
     * AI Provider instance
     *
     * @var    AIProviderInterface
     * @since  1.0.0
     */
    private AIProviderInterface $aiProvider;

    /**
     * Action Registry instance
     *
     * @var    ActionRegistry
     * @since  1.0.0
     */
    private ActionRegistry $actionRegistry;

    /**
     * Action Logger instance
     *
     * @var    ActionLogger
     * @since  1.0.0
     */
    private ActionLogger $logger;

    /**
     * Plugin parameters
     *
     * @var    array
     * @since  1.0.0
     */
    private array $params;

    /**
     * Conversation history for context
     *
     * @var    array
     * @since  1.0.0
     */
    private array $conversationHistory = [];

    /**
     * Maximum iterations for agent loop (reduced to prevent token limit errors)
     *
     * @var    int
     * @since  1.0.0
     */
    private int $maxIterations = 12;  // Reduced from 20 to avoid hitting token limits

    /**
     * Maximum conversation history size (aggressive reduction to prevent token limit errors)
     *
     * @var    int
     * @since  1.0.0
     */
    private int $maxHistorySize = 6;  // Very aggressive: system prompt + last 5 messages only

    /**
     * Constructor
     *
     * @param   AIProviderInterface  $aiProvider      AI provider instance
     * @param   ActionRegistry       $actionRegistry  Action registry instance
     * @param   ActionLogger         $logger          Logger instance
     * @param   array                $params          Plugin parameters
     *
     * @since   1.0.0
     */
    public function __construct(
        AIProviderInterface $aiProvider,
        ActionRegistry $actionRegistry,
        ActionLogger $logger,
        array $params
    ) {
        $this->aiProvider = $aiProvider;
        $this->actionRegistry = $actionRegistry;
        $this->logger = $logger;
        $this->params = $params;
        $this->maxIterations = (int) ($params['max_actions_per_request'] ?? 10);
    }

    /**
     * Process a user prompt through the agentic reasoning loop
     *
     * @param   string   $userPrompt    User's natural language request
     * @param   boolean  $requireReview Whether to require review before execution
     * @param   boolean  $streaming     Whether to stream responses in real-time
     *
     * @return  array    Result containing actions taken, summary, and status
     * @since   1.0.0
     */
    public function processPrompt(string $userPrompt, bool $requireReview = true, bool $streaming = false): array
    {
        Log::add('[DEBUG] ========== processPrompt STARTED ==========', Log::DEBUG, 'aiassistant');
        Log::add('[DEBUG] Prompt: ' . substr($userPrompt, 0, 200), Log::DEBUG, 'aiassistant');
        Log::add('[DEBUG] requireReview: ' . ($requireReview ? 'true' : 'false') . ', streaming: ' . ($streaming ? 'true' : 'false'), Log::DEBUG, 'aiassistant');
        
        $sessionId = $this->logger->startSession($userPrompt);
        Log::add('[DEBUG] Session started: ' . $sessionId, Log::DEBUG, 'aiassistant');
        
        try {
            Log::add('[DEBUG] About to initialize conversation', Log::DEBUG, 'aiassistant');
            
            // Initialize the conversation with system prompt and user request
            $this->initializeConversation($userPrompt);
            
            Log::add('[DEBUG] Conversation initialized successfully, history size: ' . count($this->conversationHistory), Log::DEBUG, 'aiassistant');

            $iteration = 0;
            $allActions = [];
            $isComplete = false;
            $executedActionSignatures = []; // Track what we've already done to prevent duplicates
            $executedActionResults = []; // Store results by signature for reuse
            $iterationThoughts = []; // Capture AI's thinking at each iteration for display
            $askedForFinalAnswer = false; // Track if we've explicitly requested a final answer

            Log::add('[DEBUG] Entering main loop, maxIterations: ' . $this->maxIterations, Log::DEBUG, 'aiassistant');

            while ($iteration < $this->maxIterations && !$isComplete) {
                $iteration++;

                Log::add('[DEBUG] ===== Starting iteration ' . $iteration . ' =====', Log::DEBUG, 'aiassistant');

                // Get AI's next thought/action
                $response = $this->aiProvider->chat($this->conversationHistory);
                
                Log::add('[DEBUG] AI Response (first 500 chars): ' . substr($response, 0, 500), Log::DEBUG, 'aiassistant');
                
                // Add assistant response to history
                $this->conversationHistory[] = [
                    'role' => 'assistant',
                    'content' => $response
                ];

                // Parse the response to extract actions and thought
                $parsedActions = $this->parseAgentResponse($response);
                
                // Extract the "thought" from the parsed response for display
                $currentThought = $this->extractThought($response);
                if ($currentThought) {
                    $thoughtData = [
                        'iteration' => $iteration,
                        'thought' => $currentThought,
                        'action_count' => count($parsedActions)
                    ];
                    $iterationThoughts[] = $thoughtData;
                    
                    // Send iteration with actual thought to display
                    if ($streaming) {
                        StreamingResponse::sendIterationStart($iteration, $currentThought);
                    }
                }
                
                Log::add('[DEBUG] Parsed ' . count($parsedActions) . ' actions from response', Log::DEBUG, 'aiassistant');

                if (empty($parsedActions)) {
                    // No actions found - check if AI provided a final answer or is being lazy
                    $hasCompleteMarker = (stripos($response, 'COMPLETE') !== false);
                    $hasSubstantiveContent = (strlen(trim($response)) > 200); // More than just a short message
                    
                    if ($hasCompleteMarker && $hasSubstantiveContent) {
                        // AI has provided final answer
                        Log::add('[DEBUG] No actions found, AI provided final answer', Log::DEBUG, 'aiassistant');
                        $isComplete = true;
                        break;
                    } else {
                        // AI is being lazy or wrote an essay without actions
                        Log::add('[WARN] AI response has no actions and no complete answer - forcing to act', Log::WARNING, 'aiassistant');
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "ERROR: You must either:\n1. Output JSON actions to get more data\n2. OR provide your FINAL ANSWER and say COMPLETE\n\nDo NOT write essays or plans. Just act!"
                        ];
                        continue;
                    }
                }

                // LIMIT to maximum 1 action per iteration to force true back-and-forth
                $maxActionsPerIteration = 1;
                if (count($parsedActions) > $maxActionsPerIteration) {
                    Log::add('[LIMIT] AI requested ' . count($parsedActions) . ' actions, limiting to ' . $maxActionsPerIteration . ' for back-and-forth flow', Log::INFO, 'aiassistant');
                    $parsedActions = array_slice($parsedActions, 0, $maxActionsPerIteration);
                }
                
                // Classify actions as read-only or write (and deduplicate)
                $readActions = [];
                $writeActions = [];
                
                foreach ($parsedActions as $action) {
                    // Create a signature for deduplication
                    $signature = $action['type'] . ':' . json_encode($action['parameters']);
                    
                    // Check if we've already executed this exact action
                    if (isset($executedActionResults[$signature])) {
                        Log::add('[DEBUG] Reusing result from duplicate action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        // Reuse the previous result and add to readActions so it gets fed back to AI
                        $cachedAction = $executedActionResults[$signature];
                        $readActions[] = $cachedAction;
                        continue;
                    }
                    
                    if ($this->isReadOnlyAction($action['type'])) {
                        $readActions[] = $action;
                        $executedActionSignatures[] = $signature;
                    } else {
                        $writeActions[] = $action;
                        $executedActionSignatures[] = $signature;
                    }
                }
                
                // Execute read-only actions immediately (gathering information)
                foreach ($readActions as $index => $action) {
                    // Skip if this is a cached result (it's already executed)
                    if (isset($action['result'])) {
                        Log::add('[DEBUG] Skipping execution of cached action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        continue;
                    }
                    
                    Log::add('[DEBUG] Auto-executing read action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                    
                    // Stream action start in real-time
                    if ($streaming) {
                        StreamingResponse::sendActionExecuting($action['type'], $action['parameters']);
                    }
                    
                    $action = $this->executeAction($action);
                    $actionId = $this->logger->logAction($sessionId, $action);
                    $action['id'] = $actionId;
                    $allActions[] = $action;
                    
                    // Cache this result for deduplication
                    $signature = $action['type'] . ':' . json_encode($action['parameters'] ?? []);
                    $executedActionResults[$signature] = $action;
                    
                    // Stream action result in real-time
                    if ($streaming) {
                        StreamingResponse::sendActionResult($action);
                    }
                    
                    // Smart follow-up: If we just queried #__menu and found SP Page Builder, auto-fetch that page
                    if ($action['type'] === 'query_database' && isset($action['result']['results'])) {
                        $followUpActions = $this->analyzeQueryForFollowUp($action['result']['results']);
                        
                        foreach ($followUpActions as $followUp) {
                            Log::add('[DEBUG] Auto-executing follow-up: ' . $followUp['type'], Log::DEBUG, 'aiassistant');
                            $followUp = $this->executeAction($followUp);
                            $followUpId = $this->logger->logAction($sessionId, $followUp);
                            $followUp['id'] = $followUpId;
                            $allActions[] = $followUp;
                            $readActions[] = $followUp;
                        }
                    }
                }
                
                // Handle write actions based on review setting
                foreach ($writeActions as $action) {
                    if ($requireReview) {
                        Log::add('[DEBUG] Queueing write action for review: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        $action['status'] = 'pending_review';
                        $this->logger->logAction($sessionId, $action);
                    } else {
                        Log::add('[DEBUG] Auto-executing write action: ' . $action['type'], Log::DEBUG, 'aiassistant');
                        $action = $this->executeAction($action);
                        $actionId = $this->logger->logAction($sessionId, $action);
                        $action['id'] = $actionId;
                    }
                    $allActions[] = $action;
                }

                // Feed execution results back to AI for read actions
                if (!empty($readActions)) {
                    $executionSummary = $this->generateExecutionSummary($readActions);
                    
                    // Check if any actions failed
                    $failedActions = array_filter($readActions, function($a) {
                        return isset($a['status']) && $a['status'] === 'failed';
                    });
                    
                    // Only force answer if we're approaching max iterations (safety net)
                    if ($iteration >= ($this->maxIterations - 2) && !$askedForFinalAnswer) {
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== EXECUTION RESULTS =====\n\n" . $executionSummary . "\n\n===== APPROACHING ITERATION LIMIT =====\nYou have " . ($this->maxIterations - $iteration) . " iterations remaining.\n\n>> PROVIDE YOUR FINAL ANSWER NOW with the data you've collected.\n\nUser's question: \"{$userPrompt}\"\n\n>> CRITICAL: Use the EXACT DATA from the results above. DO NOT make up generic content!\n>> If you queried database tables, show the ACTUAL rows/data returned.\n>> Be specific - use real titles, IDs, and values from the results.\n\n>> Synthesize the SPECIFIC data you collected\n>> Write a complete, detailed answer using REAL data\n>> Do NOT hallucinate or guess - use what you actually retrieved!\n\n-> Write your answer using the exact data, THEN say COMPLETE"
                        ];
                        $askedForFinalAnswer = true;
                    } elseif (!empty($failedActions)) {
                        // If actions failed, explicitly guide retry
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== RESULTS =====\n\n" . $executionSummary . "\n\n===== FAILED ACTIONS DETECTED =====\n\n" .
                                        "Some of your queries FAILED. You must analyze the error messages and RETRY with corrections.\n\n" .
                                        "Common fixes:\n" .
                                        "- SQL syntax error? Check for MySQL reserved words like `fulltext`, `key`, `order` - use backticks!\n" .
                                        "- Table doesn't exist? Use list_tables to discover the correct table name\n" .
                                        "- Column doesn't exist? Use describe_table to see available columns\n\n" .
                                        "REQUEST 1 CORRECTED ACTION NOW (JSON format only)"
                        ];
                    } else {
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => "===== RESULTS =====\n\n" . $executionSummary . "\n\n" .
                                        "===== INFORMATION SUFFICIENCY CHECK =====\n\n" .
                                        "User's question: \"{$userPrompt}\"\n\n" .
                                        "Before you can answer, verify you have ALL necessary information:\n\n" .
                                        "1. Do you have the specific data needed to answer this question?\n" .
                                        "2. If the question asks for \"all\" or \"each\" - do you have ALL items, not just some?\n" .
                                        "3. If the question asks for \"descriptions\" - do you have actual content, not just titles/metadata?\n" .
                                        "4. Are there any failed queries that prevented you from getting critical data?\n\n" .
                                        "===== YOUR RESPONSE =====\n\n" .
                                        ">> IF YOU HAVE ALL INFORMATION:\n" .
                                        "   Write a COMPLETE, DETAILED answer using the EXACT data above\n" .
                                        "   Then say COMPLETE\n\n" .
                                        ">> IF YOU'RE MISSING INFORMATION:\n" .
                                        "   Request 1 MORE action to gather the missing data (JSON format only)\n\n" .
                                        "RULES:\n" .
                                        "- EXACTLY 1 action per turn\n" .
                                        "- Use REAL data from results, don't make up content!\n" .
                                        "- Take your time - you have up to " . $this->maxIterations . " iterations total"
                        ];
                    }
                    
                    Log::add('[DEBUG] Fed results back to AI, continuing conversation (iteration ' . $iteration . '), askedForFinalAnswer=' . ($askedForFinalAnswer ? 'true' : 'false'), Log::DEBUG, 'aiassistant');
                    
                    // Trim conversation history if it gets too large
                    $this->trimConversationHistory();
                    
                    // Don't mark as complete yet - let AI process the results
                    continue;
                }
                
                // For write actions with review, mark as complete
                if ($requireReview && !empty($writeActions)) {
                    $isComplete = true;
                }

                // Check if agent says it's complete
                // But ONLY if we've actually done some work (gathered info or taken actions)
                if ((stripos($response, 'COMPLETE') !== false || stripos($response, 'task is finished') !== false) && !empty($allActions)) {
                    // Check if the response contains a proper answer (not just COMPLETE)
                    $responseWithoutJson = preg_replace('/```json.*?```/s', '', $response);
                    $responseWithoutJson = trim(str_replace('COMPLETE', '', $responseWithoutJson));
                    
                    // Quality checks for completeness
                    $answerLength = strlen($responseWithoutJson);
                    $endsIncomplete = preg_match('/\b(featuring|including|such as|like|with|for|and|or|but|the|a|an)\s*$/i', $responseWithoutJson);
                    
                    // For "list" or "describe" queries, expect substantial content
                    $needsDetailedAnswer = (stripos($userPrompt, 'list') !== false || 
                                          stripos($userPrompt, 'describe') !== false || 
                                          stripos($userPrompt, 'each') !== false ||
                                          stripos($userPrompt, 'all') !== false);
                    $minLength = $needsDetailedAnswer ? 200 : 100;
                    
                    if ($answerLength > $minLength && !$endsIncomplete) {
                        // Good - AI provided a complete answer
                        Log::add('[COMPLETE-CHECK] Answer is complete (' . $answerLength . ' chars)', Log::INFO, 'aiassistant');
                        $isComplete = true;
                    } else {
                        // AI's answer is too short or incomplete - force a better one
                        $reason = $endsIncomplete ? 'ends mid-sentence' : 'too short (' . $answerLength . ' < ' . $minLength . ' chars)';
                        Log::add('[WARN] AI said COMPLETE but answer is ' . $reason . ', forcing final answer', Log::WARNING, 'aiassistant');
                        
                        $executionSummary = $this->generateExecutionSummary($allActions);
                        
                        $this->conversationHistory[] = [
                            'role' => 'user',
                            'content' => ">> Your answer is INCOMPLETE (reason: {$reason})!\n\n" .
                                        "User's question: \"{$userPrompt}\"\n\n" .
                                        "All the data you collected:\n\n{$executionSummary}\n\n" .
                                        "===== YOUR TASK =====\n" .
                                        "Provide a COMPLETE, DETAILED answer using ALL the data above.\n" .
                                        "- Use REAL data from the results (titles, IDs, actual content)\n" .
                                        "- Be specific and thorough\n" .
                                        "- Format it nicely for the user\n" .
                                        "- DON'T cut off mid-sentence!\n" .
                                        "- For 'list all X', include ALL items, not just one\n\n" .
                                        "Write your FULL answer NOW, then say COMPLETE:"
                        ];
                        continue;
                    }
                } elseif ((stripos($response, 'COMPLETE') !== false || stripos($response, 'task is finished') !== false) && empty($allActions)) {
                    // AI is trying to complete without doing anything - warn and continue
                    Log::add('[WARN] AI tried to complete without taking any actions, forcing continuation', Log::WARNING, 'aiassistant');
                    $this->conversationHistory[] = [
                        'role' => 'user',
                        'content' => "You haven't gathered any information yet! Please use the available actions (query_database, list_tables, describe_table) to gather the information needed to answer the user's question."
                    ];
                    continue;
                }
            }

            // Get the last AI response for informational queries
            $lastAiResponse = '';
            foreach (array_reverse($this->conversationHistory) as $msg) {
                if ($msg['role'] === 'assistant') {
                    $lastAiResponse = $msg['content'];
                    break;
                }
            }
            
            // Extract clean answer (remove JSON blocks if present)
            $cleanAnswer = $this->extractCleanAnswer($lastAiResponse);
            
            // CRITICAL: Check if we have failed actions that need retry
            $hasFailedActions = false;
            foreach ($allActions as $action) {
                if (isset($action['status']) && $action['status'] === 'failed') {
                    $hasFailedActions = true;
                    break;
                }
            }
            
            // If we have failed actions, don't force answer yet - AI should retry
            if ($hasFailedActions && !$isComplete) {
                Log::add('[FORCE-ANSWER] Skipping - there are failed actions that may need retry', Log::INFO, 'aiassistant');
            }
            // If we have actions but no substantive answer, force one final synthesis
            elseif (!empty($allActions) && strlen($cleanAnswer) < 100) {
                Log::add('[FORCE-ANSWER] AI completed actions but provided no answer. Forcing final synthesis.', Log::WARNING, 'aiassistant');
                
                // Build a summary of all the data collected
                $allResults = $this->generateExecutionSummary($allActions);
                
                // Count successful actions
                $successfulActions = array_filter($allActions, function($a) {
                    return isset($a['status']) && $a['status'] === 'executed';
                });
                
                $this->conversationHistory[] = [
                    'role' => 'user',
                    'content' => "===== FINAL SYNTHESIS REQUIRED =====\n\n" .
                                "You have completed " . count($successfulActions) . " successful actions (out of " . count($allActions) . " total).\n\n" .
                                "User's original question: \"{$userPrompt}\"\n\n" .
                                "All the data you collected:\n\n{$allResults}\n\n" .
                                "===== YOUR TASK =====\n" .
                                "Using the EXACT data above, provide a COMPLETE answer to the user's question.\n" .
                                ">> If some queries failed and you're missing critical data, SAY SO - don't make up content!\n" .
                                "- Use REAL data from the successful results (IDs, titles, values)\n" .
                                "- Be specific and detailed\n" .
                                "- Format it nicely for the user\n" .
                                "- Do NOT request more actions\n\n" .
                                "Write your final answer NOW:"
                ];
                
                // Get final synthesis
                $finalResponse = $this->aiProvider->chat($this->conversationHistory);
                $this->conversationHistory[] = [
                    'role' => 'assistant',
                    'content' => $finalResponse
                ];
                
                $cleanAnswer = $this->extractCleanAnswer($finalResponse);
                Log::add('[FORCE-ANSWER] Received final answer (' . strlen($cleanAnswer) . ' chars)', Log::INFO, 'aiassistant');
            }
            
            $result = [
                'success' => true,
                'session_id' => $sessionId,
                'actions' => $allActions,
                'iterations' => $iteration,
                'iteration_thoughts' => $iterationThoughts,  // NEW: Show thinking at each step
                'status' => $requireReview ? 'pending_review' : 'executed',
                'summary' => $this->generateFinalSummary($allActions, $isComplete),
                'conversation' => $this->conversationHistory,
                'ai_response' => $cleanAnswer  // Clean AI response without JSON
            ];

            $this->logger->endSession($sessionId, $result);

            // Stream final completion
            if ($streaming) {
                StreamingResponse::sendComplete($result);
            }

            return $result;

        } catch (\Exception $e) {
            // Log detailed error internally
            $errorMsg = 'AI Assistant session error: ' . $e->getMessage() . 
                       ' | File: ' . $e->getFile() . 
                       ' | Line: ' . $e->getLine() . 
                       ' | Trace: ' . $e->getTraceAsString();
            
            Log::add('[EXCEPTION] ' . $errorMsg, Log::ERROR, 'aiassistant');

            $publicError = $e->getMessage() . ' (in ' . basename($e->getFile()) . ' line ' . $e->getLine() . ')';

            $error = [
                'success' => false,
                'session_id' => $sessionId,
                'error' => $publicError
            ];

            $this->logger->logError($sessionId, $error);

            // Stream error if streaming is enabled
            if ($streaming) {
                StreamingResponse::sendError($publicError);
            }

            return $error;
        } catch (\Throwable $t) {
            // Catch any fatal errors (PHP 7+)
            $errorMsg = 'AI Assistant FATAL error: ' . $t->getMessage() . 
                       ' | File: ' . $t->getFile() . 
                       ' | Line: ' . $t->getLine() . 
                       ' | Trace: ' . $t->getTraceAsString();
            
            Log::add('[FATAL] ' . $errorMsg, Log::ERROR, 'aiassistant');

            $publicError = $t->getMessage() . ' (in ' . basename($t->getFile()) . ' line ' . $t->getLine() . ')';

            $error = [
                'success' => false,
                'session_id' => $sessionId ?? 'unknown',
                'error' => $publicError
            ];

            // Stream error if streaming is enabled
            if ($streaming) {
                StreamingResponse::sendError($publicError);
            }

            return $error;
        }
    }

    /**
     * Execute previously planned actions after review
     *
     * @param   string  $sessionId  Session ID from planning phase
     * @param   array   $actionIds  Array of action IDs to execute
     *
     * @return  array   Execution results
     * @since   1.0.0
     */
    public function executeReviewedActions(string $sessionId, array $actionIds = []): array
    {
        $actions = $this->logger->getSessionActions($sessionId);
        $results = [];

        foreach ($actions as $action) {
            // If specific IDs provided, only execute those
            if (!empty($actionIds) && !in_array($action['id'], $actionIds)) {
                continue;
            }

            if ($action['status'] === 'pending_review') {
                $executedAction = $this->executeAction($action);
                $results[] = $executedAction;
                $this->logger->updateAction($action['id'], $executedAction);
            }
        }

        return [
            'success' => true,
            'executed_actions' => $results,
            'summary' => $this->generateExecutionSummary($results)
        ];
    }

    /**
     * Generate a compact database schema map for the AI
     *
     * @return  string  Formatted schema information
     * @since   1.0.0
     */
    private function generateDatabaseSchema(): string
    {
        $db = \Joomla\CMS\Factory::getDbo();
        $prefix = $db->getPrefix();
        
        try {
            // Get all tables
            $tables = $db->getTableList();
            
            // Key tables to describe in detail
            $keyTables = [
                '#__menu' => ['id', 'menutype', 'title', 'alias', 'link', 'type', 'published', 'home', 'client_id'],
                '#__content' => ['id', 'title', 'alias', 'introtext', 'fulltext', 'state', 'catid', 'created', 'featured'],
                '#__categories' => ['id', 'title', 'alias', 'extension', 'published', 'parent_id'],
                '#__modules' => ['id', 'title', 'module', 'position', 'published', 'ordering'],
                '#__extensions' => ['extension_id', 'name', 'type', 'element', 'enabled']
            ];
            
            $schema = "\n## DATABASE SCHEMA:\n";
            
            // Show compact schema for key tables only
            foreach ($keyTables as $tableName => $columns) {
                $fullTableName = str_replace('#__', $prefix, $tableName);
                if (in_array($fullTableName, $tables)) {
                    $schema .= $tableName . ": " . implode(', ', $columns) . "\n";
                }
            }
            
            // Group remaining tables by component (very compact)
            $groupedTables = [];
            foreach ($tables as $fullTable) {
                $tableName = str_replace($prefix, '#__', $fullTable);
                if (isset($keyTables[$tableName])) {
                    continue;
                }
                if (preg_match('/#__([^_]+)_/', $tableName, $matches)) {
                    $component = $matches[1];
                    if (!isset($groupedTables[$component])) {
                        $groupedTables[$component] = 0;
                    }
                    $groupedTables[$component]++;
                }
            }
            
            // Show counts only for installed components
            $schema .= "\nComponents: ";
            ksort($groupedTables);
            $compactList = [];
            foreach ($groupedTables as $component => $count) {
                $compactList[] = "{$component}({$count})";
            }
            $schema .= implode(', ', array_slice($compactList, 0, 15));
            if (count($compactList) > 15) {
                $schema .= " +" . (count($compactList) - 15) . " more";
            }
            $schema .= "\nPrefix: #__\n";
            
            return $schema;
            
        } catch (\Exception $e) {
            return "\n## DATABASE SCHEMA:\nUse `list_tables` and `describe_table` to discover schema.\n";
        }
    }

    /**
     * Initialize conversation with system prompt and context
     *
     * @param   string  $userPrompt  User's request
     *
     * @return  void
     * @since   1.0.0
     */
    private function initializeConversation(string $userPrompt): void
    {
        $availableActions = $this->actionRegistry->getAvailableActions();
        $actionDescriptions = [];

        foreach ($availableActions as $actionName => $actionClass) {
            $actionDescriptions[] = "- {$actionName}: " . $actionClass::getDescription();
        }
        
        // Generate live database schema
        $databaseSchema = $this->generateDatabaseSchema();

        $systemPrompt = <<<SYSTEM
You are an AI assistant with FULL ACCESS to a Joomla 5 CMS. You can explore and query ANY data in the system.

>> YOU ARE NOT LIMITED TO PREDEFINED ACTIONS!

You have dynamic database access and can discover everything on your own:

{$this->formatActionList($availableActions)}

## YOUR CAPABILITIES:

### 1. **QUERY THE DATABASE** (Your main tool):
- `query_database` - Execute ANY SELECT query on ANY table (see schema above!)
- `describe_table` - Get detailed column info if needed (e.g., for tables not shown above)
- `list_extensions` - Find installed components/modules/plugins details

{$databaseSchema}

### 2. PARSE MENU LINKS:
`#__menu.link` shows component & ID:
- `option=com_sppagebuilder&id=9` → query `#__sppagebuilder WHERE id=9`
- `option=com_content&id=15` → query `#__content WHERE id=15`

### 3. APPROACH:
- Parse menu links to understand what component each page uses
- Query the actual content tables to get real data
- Use exact data from queries (no assumptions or placeholders)

### 4. CORE RULES:
- Unknown table columns? Use `describe_table` to discover them
- Query failed? Read the error, fix it, retry
- Use backticks for SQL reserved words: `fulltext`, `order`, `key`
- Format your answer well (markdown, structure, readability)
- Don't answer until you have complete information

## WORKFLOW:
1. Output 1 JSON action (EXACTLY ONE!)
2. Get result → analyze
3. Have all data? → Answer + COMPLETE
4. Missing data? → Request 1 more action

## RULES:
- ONE action per turn (system enforces this)
- See result before next action
- Use EXACT data from queries (no placeholders/examples!)
- Answer only when you have complete info

## ACTION FORMAT (CRITICAL):

**Your "thought" field is shown to the user**, so make it clear and helpful!

```json
{
  "thought": "Need to find the homepage menu item to see what content it links to",
  "actions": [
    {"type": "query_database", "parameters": {"query": "SELECT * FROM #__menu WHERE home=1"}, "reason": "Get homepage menu"}
  ]
}
```

Site: {$this->getSiteName()} | Joomla {JVERSION} | {$this->getSiteUrl()}

**EXPLORE. DISCOVER. SOLVE.**
SYSTEM;

        $this->conversationHistory = [
            ['role' => 'system', 'content' => $systemPrompt],
            ['role' => 'user', 'content' => $userPrompt]
        ];
    }

    /**
     * Parse agent response to extract structured actions
     *
     * @param   string  $response  AI response text
     *
     * @return  array   Array of parsed actions
     * @since   1.0.0
     */
    private function parseAgentResponse(string $response): array
    {
        $actions = [];
        $jsonData = null;

        // Method 1: Try to extract JSON from code blocks first
        if (preg_match('/```json\s*(\{(?:[^`]|`(?!``))*\})\s*```/s', $response, $matches)) {
            $jsonData = json_decode($matches[1], true);
            Log::add('[DEBUG] Found JSON in code block', Log::DEBUG, 'aiassistant');
        }
        
        // Method 2: Try to find raw JSON object in the response
        if (!$jsonData && preg_match('/\{[^{}]*"thought"[^{}]*"actions"[^{}]*\[[^\]]*\][^{}]*\}/s', $response, $matches)) {
            $jsonData = json_decode($matches[0], true);
            Log::add('[DEBUG] Found raw JSON object', Log::DEBUG, 'aiassistant');
        }
        
        // Method 3: More aggressive - find any valid JSON with "actions" key
        if (!$jsonData) {
            // Look for balanced braces that contain "actions"
            if (preg_match('/(\{(?:[^{}]|(?R))*\})/s', $response, $matches)) {
                $possibleJson = $matches[1];
                if (strpos($possibleJson, '"actions"') !== false) {
                    $jsonData = json_decode($possibleJson, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        Log::add('[DEBUG] Found JSON via aggressive search', Log::DEBUG, 'aiassistant');
                    } else {
                        $jsonData = null;
                    }
                }
            }
        }
        
        if ($jsonData && json_last_error() === JSON_ERROR_NONE && isset($jsonData['actions']) && is_array($jsonData['actions'])) {
            Log::add('[DEBUG] Successfully parsed JSON, found ' . count($jsonData['actions']) . ' actions', Log::DEBUG, 'aiassistant');
            
            foreach ($jsonData['actions'] as $action) {
                if (isset($action['type'])) {
                    $actions[] = [
                        'type' => $this->sanitizeString($action['type']),
                        'parameters' => $this->sanitizeParameters($action['parameters'] ?? []),
                        'reason' => $this->sanitizeString($action['reason'] ?? ''),
                        'thought' => $this->sanitizeString($jsonData['thought'] ?? ''),
                        'status' => 'planned'
                    ];
                }
            }
        } else {
            Log::add('[DEBUG] Could not parse actions from response. Response (first 500 chars): ' . substr($response, 0, 500), Log::DEBUG, 'aiassistant');
        }

        return $actions;
    }

    /**
     * Sanitize a string parameter
     *
     * @param   mixed  $value  Value to sanitize
     *
     * @return  string  Sanitized string
     * @since   1.0.0
     */
    private function sanitizeString($value): string
    {
        if (!is_string($value)) {
            return '';
        }
        // Remove any null bytes and trim
        return trim(str_replace("\0", '', $value));
    }

    /**
     * Sanitize parameters array
     *
     * @param   array  $parameters  Parameters to sanitize
     *
     * @return  array  Sanitized parameters
     * @since   1.0.0
     */
    private function sanitizeParameters(array $parameters): array
    {
        $sanitized = [];
        
        foreach ($parameters as $key => $value) {
            // Sanitize key
            $cleanKey = preg_replace('/[^a-zA-Z0-9_]/', '', $key);
            
            if (is_array($value)) {
                $sanitized[$cleanKey] = $this->sanitizeParameters($value);
            } elseif (is_string($value)) {
                $sanitized[$cleanKey] = $this->sanitizeString($value);
            } elseif (is_numeric($value) || is_bool($value)) {
                $sanitized[$cleanKey] = $value;
            }
        }
        
        return $sanitized;
    }

    /**
     * Execute a single action
     *
     * @param   array  $action  Action definition
     *
     * @return  array  Action with execution result
     * @since   1.0.0
     */
    private function executeAction(array $action): array
    {
        try {
            $actionHandler = $this->actionRegistry->getAction($action['type']);
            
            if (!$actionHandler) {
                throw new \RuntimeException("Unknown action type: {$action['type']}");
            }

            $result = $actionHandler->execute($action['parameters']);
            
            $action['status'] = 'executed';
            $action['result'] = $result;
            $action['executed_at'] = date('Y-m-d H:i:s');
            
            // Store before_state if available (for undo functionality)
            if (isset($result['before_state'])) {
                $action['before_state'] = $result['before_state'];
                unset($result['before_state']); // Don't show in result
                $action['result'] = $result;
            }

        } catch (\Exception $e) {
            $action['status'] = 'failed';
            $action['error'] = $e->getMessage();
        }

        return $action;
    }

    /**
     * Generate execution summary for feedback to AI
     *
     * @param   array  $actions  Array of actions
     *
     * @return  string  Summary text
     * @since   1.0.0
     */
    private function generateExecutionSummary(array $actions): string
    {
        $summary = [];

        foreach ($actions as $action) {
            $status = $action['status'] ?? 'unknown';
            $type = $action['type'];
            
            if ($status === 'executed') {
                // Format the result in a readable way for the AI
                $result = $action['result'];
                if (is_array($result)) {
                    // Pretty print for better AI understanding
                    $resultText = json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
                } else {
                    $resultText = (string) $result;
                }
                $summary[] = "Action: {$type}\nStatus: Success\nResult:\n{$resultText}\n";
            } elseif ($status === 'failed') {
                $summary[] = "Action: {$type}\nStatus: Failed\nError: {$action['error']}\n";
            }
        }

        return implode("\n---\n", $summary);
    }

    /**
     * Generate final summary for user
     *
     * @param   array    $actions     All actions taken
     * @param   boolean  $isComplete  Whether task completed successfully
     *
     * @return  string   Human-readable summary
     * @since   1.0.0
     */
    private function generateFinalSummary(array $actions, bool $isComplete): string
    {
        $total = count($actions);
        $executed = count(array_filter($actions, fn($a) => $a['status'] === 'executed'));
        $failed = count(array_filter($actions, fn($a) => $a['status'] === 'failed'));
        $pending = count(array_filter($actions, fn($a) => $a['status'] === 'pending_review'));

        $summary = "Task " . ($isComplete ? "completed" : "processed") . ".\n\n";
        $summary .= "Actions: {$total} total";
        
        if ($executed > 0) {
            $summary .= ", {$executed} executed";
        }
        if ($failed > 0) {
            $summary .= ", {$failed} failed";
        }
        if ($pending > 0) {
            $summary .= ", {$pending} pending review";
        }

        return $summary;
    }

    /**
     * Format action list for system prompt
     *
     * @param   array  $actions  Available actions
     *
     * @return  string  Formatted list
     * @since   1.0.0
     */
    private function formatActionList(array $actions): string
    {
        $list = [];
        
        foreach ($actions as $name => $class) {
            $list[] = "- **{$name}**: " . $class::getDescription();
        }

        return implode("\n", $list);
    }

    /**
     * Get site name
     *
     * @return  string
     * @since   1.0.0
     */
    private function getSiteName(): string
    {
        $app = Factory::getApplication();
        return $app ? $app->get('sitename', 'Joomla Site') : 'Joomla Site';
    }

    /**
     * Get site URL
     *
     * @return  string
     * @since   1.0.0
     */
    private function getSiteUrl(): string
    {
        return Uri::root();
    }

    /**
     * Check if action is read-only (information gathering) vs write (modifying)
     *
     * @param   string  $actionType  Type of action
     *
     * @return  boolean  True if read-only
     * @since   1.0.0
     */
    private function isReadOnlyAction(string $actionType): bool
    {
        $readOnlyActions = [
            // Universal discovery actions (work on ANY Joomla site)
            'query_database',
            'list_tables',
            'describe_table',
            'list_extensions',
            'get_site_info'
        ];
        
        return in_array($actionType, $readOnlyActions);
    }

    /**
     * Consolidate conversation history intelligently to save tokens
     * 
     * Instead of just truncating, we:
     * 1. Summarize old action results into key facts
     * 2. Remove redundant queries
     * 3. Preserve critical data (IDs, titles, counts)
     * 4. Keep recent messages in full
     *
     * @return  void
     * @since   1.0.0
     */
    private function trimConversationHistory(): void
    {
        if (count($this->conversationHistory) <= $this->maxHistorySize) {
            return; // No trimming needed
        }
        
        // Always keep system message
        $systemMessage = $this->conversationHistory[0];
        
        // Keep last 3 messages in full (most recent context)
        $keepFullCount = 3;
        $recentMessages = array_slice($this->conversationHistory, -$keepFullCount);
        
        // Consolidate middle messages (between system and recent)
        $middleMessages = array_slice($this->conversationHistory, 1, count($this->conversationHistory) - $keepFullCount - 1);
        $consolidatedMiddle = $this->consolidateMessages($middleMessages);
        
        // Rebuild history: system + consolidated + recent full messages
        $this->conversationHistory = array_merge(
            [$systemMessage],
            $consolidatedMiddle,
            $recentMessages
        );
    }
    
    /**
     * Consolidate messages into a compact summary
     * 
     * Extracts key facts from action results and removes verbosity
     *
     * @param   array  $messages  Messages to consolidate
     *
     * @return  array  Consolidated messages
     * @since   1.0.0
     */
    private function consolidateMessages(array $messages): array
    {
        if (empty($messages)) {
            return [];
        }
        
        $keyFacts = [];
        $queriesSeen = [];
        
        foreach ($messages as $message) {
            if (!isset($message['content'])) {
                continue;
            }
            
            $content = $message['content'];
            
            // Skip if duplicate query (extract table name)
            if (preg_match('/FROM\s+(#__\w+)/i', $content, $matches)) {
                $table = $matches[1];
                if (isset($queriesSeen[$table])) {
                    continue; // Skip duplicate query on same table
                }
                $queriesSeen[$table] = true;
            }
            
            // Extract key facts from action results
            if ($message['role'] === 'user' && strpos($content, 'Action Result:') !== false) {
                // Try to parse JSON result
                if (preg_match('/\{[\s\S]*\}/', $content, $jsonMatch)) {
                    try {
                        $result = json_decode($jsonMatch[0], true);
                        
                        if (isset($result['count']) && isset($result['results'])) {
                            // Database query result
                            $count = $result['count'];
                            $table = 'unknown';
                            if (preg_match('/FROM\s+(#__\w+)/i', $content, $m)) {
                                $table = $m[1];
                            }
                            
                            // Extract key IDs/titles if present
                            $items = [];
                            if ($count > 0 && is_array($result['results'])) {
                                foreach (array_slice($result['results'], 0, 3) as $row) {
                                    if (isset($row['id']) && isset($row['title'])) {
                                        $items[] = "#{$row['id']} {$row['title']}";
                                    } elseif (isset($row['id'])) {
                                        $items[] = "#{$row['id']}";
                                    }
                                }
                            }
                            
                            $summary = "Found {$count} rows in {$table}";
                            if (!empty($items)) {
                                $summary .= ": " . implode(', ', $items);
                                if ($count > count($items)) {
                                    $summary .= " +" . ($count - count($items)) . " more";
                                }
                            }
                            $keyFacts[] = $summary;
                        }
                    } catch (\Exception $e) {
                        // Skip if can't parse
                    }
                }
            }
        }
        
        // If we extracted key facts, return them as a single consolidated message
        if (!empty($keyFacts)) {
            return [
                [
                    'role' => 'user',
                    'content' => "Previous findings:\n- " . implode("\n- ", $keyFacts)
                ]
            ];
        }
        
        // Otherwise, just keep first and last message from this batch
        if (count($messages) > 2) {
            return [
                $messages[0],
                [
                    'role' => 'user',
                    'content' => '[' . (count($messages) - 2) . ' messages consolidated to save tokens]'
                ],
                end($messages)
            ];
        }
        
        return $messages;
    }

    /**
     * Analyze query results and generate intelligent follow-up actions
     * 
     * DISABLED: Auto-follow-up was too prescriptive and assumed component knowledge.
     * The AI should discover and query tables on its own through iterative exploration.
     *
     * @param   array  $results  Query results
     *
     * @return  array  Follow-up actions to execute
     * @since   1.0.0
     */
    private function analyzeQueryForFollowUp(array $results): array
    {
        // DISABLED: Let the AI discover and query tables itself
        return [];
        
        $followUps = [];
        
        foreach ($results as $row) {
            // Convert stdClass to array if needed
            if (is_object($row)) {
                $row = json_decode(json_encode($row), true);
            }
            
            // Check if this is a menu item
            if (isset($row['link']) && isset($row['menutype'])) {
                $link = $row['link'];
                
                // SP Page Builder page
                if (strpos($link, 'com_sppagebuilder') !== false && strpos($link, 'view=page') !== false) {
                    if (preg_match('/id=(\d+)/', $link, $matches)) {
                        $pageId = (int) $matches[1];
                        Log::add('[SMART-FOLLOW-UP] Found SP Page Builder page ID: ' . $pageId, Log::INFO, 'aiassistant');
                        
                        // SP Page Builder stores pages in #__sppagebuilder (not #__sppagebuilder_pages)
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__sppagebuilder WHERE id=" . $pageId
                            ],
                            'reason' => 'Auto-fetching SP Page Builder page from #__sppagebuilder table',
                            'status' => 'planned'
                        ];
                        
                        // Get page addons (the actual content elements)
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__sppagebuilder_addons WHERE page_id=" . $pageId . " ORDER BY ordering"
                            ],
                            'reason' => 'Getting page content addons from #__sppagebuilder_addons',
                            'status' => 'planned'
                        ];
                    }
                }
                // Single article
                elseif (strpos($link, 'com_content') !== false && strpos($link, 'view=article') !== false) {
                    if (preg_match('/id=(\d+)/', $link, $matches)) {
                        $articleId = (int) $matches[1];
                        Log::add('[SMART-FOLLOW-UP] Found article ID: ' . $articleId, Log::INFO, 'aiassistant');
                        
                        $followUps[] = [
                            'type' => 'query_database',
                            'parameters' => [
                                'query' => "SELECT * FROM #__content WHERE id=" . $articleId
                            ],
                            'reason' => 'Auto-fetching article content for article ID ' . $articleId,
                            'status' => 'planned'
                        ];
                    }
                }
                // Featured articles
                elseif (strpos($link, 'view=featured') !== false) {
                    Log::add('[SMART-FOLLOW-UP] Found featured articles layout', Log::INFO, 'aiassistant');
                    
                    $followUps[] = [
                        'type' => 'query_database',
                        'parameters' => [
                            'query' => "SELECT * FROM #__content WHERE featured=1 AND state=1 ORDER BY created DESC LIMIT 10"
                        ],
                        'reason' => 'Auto-fetching featured articles',
                        'status' => 'planned'
                    ];
                }
            }
        }
        
        return $followUps;
    }

    /**
     * Extract clean answer from AI response (remove JSON action blocks)
     *
     * @param   string  $response  Raw AI response
     *
     * @return  string  Clean answer text
     * @since   1.0.0
     */
    private function extractCleanAnswer(string $response): string
    {
        // Remove JSON code blocks
        $clean = preg_replace('/```json\s*\{[^`]*\}\s*```/s', '', $response);
        
        // Remove raw JSON objects (action plans)
        $clean = preg_replace('/\{[^{}]*"thought"[^{}]*"actions"[^{}]*\[[^\]]*\][^{}]*\}/s', '', $clean);
        
        // Remove "COMPLETE" marker
        $clean = str_ireplace('COMPLETE', '', $clean);
        
        // Clean up extra whitespace
        $clean = preg_replace('/\n{3,}/', "\n\n", $clean);
        $clean = trim($clean);
        
        // If nothing left, return a default message
        if (empty($clean)) {
            return "Actions completed. See details below.";
        }
        
        return $clean;
    }

    /**
     * Extract the "thought" field from AI's response
     *
     * @param   string  $response  AI response
     *
     * @return  string  Extracted thought or empty string
     * @since   1.0.0
     */
    private function extractThought(string $response): string
    {
        // Try to extract thought from JSON
        if (preg_match('/"thought"\s*:\s*"([^"]+)"/s', $response, $matches)) {
            return $matches[1];
        }
        
        // If no JSON thought, use the first sentence/line of the response
        $lines = explode("\n", trim($response));
        $firstLine = trim($lines[0] ?? '');
        
        // If it's not JSON and has substance, return it
        if ($firstLine && strpos($firstLine, '{') !== 0 && strlen($firstLine) > 10) {
            return substr($firstLine, 0, 200); // Limit length
        }
        
        return '';
    }

}

