<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Read Menu Action
 *
 * @since  1.0.0
 */
class ReadMenuAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Menu data
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }
        
        // If menu_type is provided, get all items from that menu
        if (!empty($parameters['menu_type'])) {
            $query = $db->getQuery(true)
                ->select('*')
                ->from($db->quoteName('#__menu'))
                ->where($db->quoteName('menutype') . ' = ' . $db->quote($parameters['menu_type']))
                ->where($db->quoteName('client_id') . ' = 0')
                ->order($db->quoteName('lft'));

            $db->setQuery($query);
            $items = $db->loadObjectList();

            return [
                'menu_type' => $parameters['menu_type'],
                'count' => count($items),
                'items' => $items
            ];
        }

        // Otherwise list all menus
        $query = $db->getQuery(true)
            ->select('DISTINCT menutype, title, description')
            ->from($db->quoteName('#__menu_types'))
            ->order($db->quoteName('menutype'));

        $db->setQuery($query);
        $menus = $db->loadObjectList();

        return [
            'count' => count($menus),
            'menus' => $menus
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Read menu information. If menu_type parameter is provided, returns all items in that menu. Otherwise, returns list of all menus.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}

