<?php
/**
 * Test OpenAI API Connection
 * DELETE THIS FILE AFTER USE
 */

define('_JEXEC', 1);
define('JPATH_BASE', __DIR__);

require_once JPATH_BASE . '/includes/defines.php';
require_once JPATH_BASE . '/includes/framework.php';

use Joomla\CMS\Factory;

$app = Factory::getApplication('administrator');
$user = $app->getIdentity();

if (!$user || !$user->authorise('core.admin')) {
    die('<h1>Unauthorized</h1>');
}

echo '<h1>OpenAI API Configuration Test</h1>';
echo '<style>pre { background: #f5f5f5; padding: 10px; border: 1px solid #ddd; } .error { color: red; } .success { color: green; } .warning { color: orange; }</style>';

// Get plugin parameters
$db = Factory::getDbo();
$query = $db->getQuery(true)
    ->select('params')
    ->from('#__extensions')
    ->where('element = ' . $db->quote('aiassistant'))
    ->where('folder = ' . $db->quote('system'));
$db->setQuery($query);
$paramsJson = $db->loadResult();

if (!$paramsJson) {
    echo '<p class="error">✗ Plugin not found in database</p>';
    die();
}

$params = json_decode($paramsJson, true);

echo '<h2>Configuration</h2>';
echo '<pre>';
echo 'AI Provider: ' . ($params['ai_provider'] ?? 'not set') . "\n";
echo 'OpenAI Model: ' . ($params['openai_model'] ?? 'not set') . "\n";
echo 'OpenAI API Key Length: ' . strlen($params['openai_api_key'] ?? '') . " characters\n";
echo 'OpenAI API Key Start: ' . substr($params['openai_api_key'] ?? '', 0, 10) . "...\n";
echo '</pre>';

// Validate API key
$apiKey = $params['openai_api_key'] ?? '';
$model = $params['openai_model'] ?? 'gpt-5';

if (empty($apiKey)) {
    echo '<p class="error">✗ OpenAI API key is not configured!</p>';
    echo '<p>Go to System → Plugins → AI Assistant and add your API key.</p>';
    die();
}

if (strlen($apiKey) < 50) {
    echo '<p class="warning">⚠ OpenAI API key seems too short (' . strlen($apiKey) . ' characters). Should be ~51 characters starting with "sk-"</p>';
}

if (!str_starts_with($apiKey, 'sk-')) {
    echo '<p class="warning">⚠ OpenAI API key should start with "sk-"</p>';
}

echo '<h2>Testing OpenAI API Connection</h2>';
echo '<p>Sending test request to OpenAI...</p>';

// Test API connection
try {
    $payload = [
        'model' => $model,
        'input' => 'Say "Hello! API is working!" in exactly those words.',
        'text' => ['verbosity' => 'low'],
        'max_output_tokens' => 100
    ];
    
    $ch = curl_init('https://api.openai.com/v1/responses');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Authorization: Bearer ' . $apiKey
    ]);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    echo '<h3>Response</h3>';
    echo '<p>HTTP Code: ' . $httpCode . '</p>';
    
    if ($error) {
        echo '<p class="error">✗ cURL Error: ' . htmlspecialchars($error) . '</p>';
    }
    
    if ($httpCode === 200) {
        echo '<p class="success">✓ API request successful!</p>';
        echo '<h4>Response Body:</h4>';
        echo '<pre>' . htmlspecialchars(substr($response, 0, 1000)) . '</pre>';
    } elseif ($httpCode === 401) {
        echo '<p class="error">✗ Authentication failed - Invalid API key</p>';
    } elseif ($httpCode === 404) {
        echo '<p class="error">✗ Endpoint not found - The model "' . htmlspecialchars($model) . '" may not exist or the API endpoint is incorrect</p>';
        echo '<p>Common models: gpt-4o, gpt-4-turbo, gpt-3.5-turbo</p>';
    } elseif ($httpCode === 429) {
        echo '<p class="error">✗ Rate limit exceeded or insufficient quota</p>';
    } else {
        echo '<p class="error">✗ API Error (HTTP ' . $httpCode . ')</p>';
        echo '<pre>' . htmlspecialchars(substr($response, 0, 500)) . '</pre>';
    }
    
} catch (\Exception $e) {
    echo '<p class="error">✗ Exception: ' . htmlspecialchars($e->getMessage()) . '</p>';
}

echo '<hr><p><em>Remember to delete this file after use!</em></p>';

