<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\Table\Table;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Read Article Action
 *
 * @since  1.0.0
 */
class ReadArticleAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Article data
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for ReadArticleAction');
        }

        $db = Factory::getContainer()->get('DatabaseDriver');
        $articleId = $parameters['article_id'];

        $query = $db->getQuery(true)
            ->select('a.*, c.title AS category_title')
            ->from($db->quoteName('#__content', 'a'))
            ->leftJoin($db->quoteName('#__categories', 'c') . ' ON a.catid = c.id')
            ->where($db->quoteName('a.id') . ' = ' . (int) $articleId);

        $db->setQuery($query);
        $article = $db->loadObject();

        if (!$article) {
            throw new \RuntimeException("Article with ID {$articleId} not found");
        }

        return [
            'id' => $article->id,
            'title' => $article->title,
            'alias' => $article->alias,
            'introtext' => $article->introtext,
            'fulltext' => $article->fulltext,
            'state' => $article->state,
            'catid' => $article->catid,
            'category_title' => $article->category_title,
            'created' => $article->created,
            'modified' => $article->modified,
            'publish_up' => $article->publish_up,
            'publish_down' => $article->publish_down,
            'images' => $article->images,
            'urls' => $article->urls,
            'metadesc' => $article->metadesc,
            'metakey' => $article->metakey,
            'metadata' => $article->metadata,
            'featured' => $article->featured,
            'language' => $article->language
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Read a single article by ID. Returns full article data including title, content, category, metadata, etc.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['article_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['article_id']) && is_numeric($parameters['article_id']);
    }
}

