<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\SubscriberInterface;

// Prevent direct access
defined('_JEXEC') or die;

/**
 * AI Assistant Plugin
 *
 * Provides an agentic AI system integrated into Joomla that can read and modify
 * site content intelligently based on natural language prompts.
 *
 * @since  1.0.0
 */
final class AiAssistant extends CMSPlugin implements SubscriberInterface
{
    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  1.0.0
     */
    protected $autoloadLanguage = true;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise'   => 'onAfterInitialise',
            'onBeforeCompileHead' => 'onBeforeCompileHead',
        ];
    }

    /**
     * After initialise event handler
     * Handles console access and AJAX requests
     *
     * @return  void
     * @since   1.0.0
     */
    public function onAfterInitialise(): void
    {
        $app = $this->getApplication();
        $input = $app->input;

        // Check if this is an AJAX request for our plugin
        if ($input->get('option') === 'com_ajax' 
            && $input->get('plugin') === 'aiassistant'
            && $input->get('group') === 'system') {
            
            // Check permissions
            $user = $app->getIdentity();
            if (!$user->authorise('core.admin', 'com_plugins')) {
                $this->sendJsonResponse(['error' => 'Unauthorized'], 403);
                return;
            }

            // Handle different request types
            $format = $input->get('format', 'json');
            
            if ($format === 'raw') {
                // Display console HTML
                $this->displayConsole();
            } else {
                // Handle AJAX API request
                $this->handleAjaxRequest();
            }
        }
    }

    /**
     * Before compile head event handler
     * Could be used to add menu items or other UI elements
     *
     * @return  void
     * @since   1.0.0
     */
    public function onBeforeCompileHead(): void
    {
        $app = $this->getApplication();

        if (!$app->isClient('administrator')) {
            return;
        }

        // Could add admin menu item here if needed
    }

    /**
     * Display the console interface
     *
     * @return  void
     * @since   1.0.0
     */
    private function displayConsole(): void
    {
        $app = $this->getApplication();
        $doc = $app->getDocument();

        // Set document type to HTML
        $doc->setType('html');

        // Load assets
        $wa = $doc->getWebAssetManager();
        
        // Register and use CSS
        $wa->registerAndUseStyle(
            'plg_system_aiassistant.console',
            'plg_system_aiassistant/console.css',
            [],
            [],
            []
        );

        // Register and use JS
        $wa->registerAndUseScript(
            'plg_system_aiassistant.console',
            'plg_system_aiassistant/console.js',
            [],
            ['defer' => true],
            []
        );

        // Include the console template
        $templatePath = JPATH_PLUGINS . '/system/aiassistant/tmpl/console.php';
        
        if (file_exists($templatePath)) {
            include $templatePath;
        } else {
            echo '<h1>AI Assistant Console</h1><p>Template not found at: ' . $templatePath . '</p>';
        }

        $app->close();
    }

    /**
     * Handle AJAX request
     *
     * @return  void
     * @since   1.0.0
     */
    private function handleAjaxRequest(): void
    {
        $app = $this->getApplication();
        $input = $app->input;

        // Validate CSRF token for POST requests
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            if (!Session::checkToken()) {
                $this->sendJsonResponse(['error' => 'Invalid token'], 403);
                return;
            }
        }

        // Apply rate limiting to all requests (both GET and POST)
        if (!$this->checkRateLimit()) {
            $this->sendJsonResponse(['error' => 'Rate limit exceeded. Please wait before making another request.'], 429);
            return;
        }

        try {
            $task = $input->get('task', 'process');

            switch ($task) {
                case 'process':
                    $this->processPrompt();
                    break;
                
                case 'execute':
                    $this->executeActions();
                    break;
                
                case 'history':
                    $this->getHistory();
                    break;
                
                case 'session':
                    $this->getSession();
                    break;
                
                default:
                    $this->sendJsonResponse(['error' => 'Unknown task'], 400);
            }

        } catch (\Exception $e) {
            // Don't expose sensitive error details
            Log::add(
                'AI Assistant error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString(),
                Log::ERROR,
                'aiassistant'
            );
            $this->sendJsonResponse(['error' => 'An error occurred while processing your request'], 500);
        }
    }

    /**
     * Process a user prompt
     *
     * @return  void
     * @since   1.0.0
     */
    private function processPrompt(): void
    {
        $app = $this->getApplication();
        $input = $app->input;

        $jsonData = file_get_contents('php://input');
        $data = json_decode($jsonData, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->sendJsonResponse(['error' => 'Invalid JSON data'], 400);
            return;
        }

        $prompt = $data['prompt'] ?? '';
        $requireReview = $data['require_review'] ?? true;

        // Validate and sanitize prompt
        if (empty($prompt) || !is_string($prompt)) {
            $this->sendJsonResponse(['error' => 'Prompt is required and must be a string'], 400);
            return;
        }

        // Limit prompt length to prevent abuse
        if (strlen($prompt) > 10000) {
            $this->sendJsonResponse(['error' => 'Prompt is too long (max 10000 characters)'], 400);
            return;
        }

        try {
            $orchestrator = $this->getOrchestrator();
            $result = $orchestrator->processPrompt($prompt, (bool) $requireReview);
            
            $this->sendJsonResponse($result);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Execute reviewed actions
     *
     * @return  void
     * @since   1.0.0
     */
    private function executeActions(): void
    {
        $app = $this->getApplication();
        $input = $app->input;

        $jsonData = file_get_contents('php://input');
        $data = json_decode($jsonData, true);

        $sessionId = $data['session_id'] ?? '';
        $actionIds = $data['action_ids'] ?? [];

        if (empty($sessionId)) {
            $this->sendJsonResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $orchestrator = $this->getOrchestrator();
            $result = $orchestrator->executeReviewedActions($sessionId, $actionIds);
            
            $this->sendJsonResponse($result);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get session history
     *
     * @return  void
     * @since   1.0.0
     */
    private function getHistory(): void
    {
        try {
            $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger(
                $this->params->get('log_all_actions', true)
            );
            $sessions = $logger->getRecentSessions(20);
            
            $this->sendJsonResponse(['sessions' => $sessions]);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get specific session details
     *
     * @return  void
     * @since   1.0.0
     */
    private function getSession(): void
    {
        $app = $this->getApplication();
        $input = $app->input;

        $sessionId = $input->get('session_id', '', 'string');

        if (empty($sessionId)) {
            $this->sendJsonResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger(
                $this->params->get('log_all_actions', true)
            );
            $actions = $logger->getSessionActions($sessionId);
            
            $this->sendJsonResponse(['actions' => $actions]);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get orchestrator instance
     *
     * @return  \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator
     * @since   1.0.0
     */
    private function getOrchestrator(): \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator
    {
        $params = $this->params->toArray();

        // Initialize AI provider
        $provider = $params['ai_provider'] ?? 'openai';
        
        if ($provider === 'anthropic') {
            $apiKey = $params['anthropic_api_key'] ?? '';
            $model = $params['anthropic_model'] ?? 'claude-3-5-sonnet-20241022';
            $aiProvider = new \Joomla\Plugin\System\AiAssistant\AI\AnthropicProvider($apiKey, $model);
        } else {
            $apiKey = $params['openai_api_key'] ?? '';
            $model = $params['openai_model'] ?? 'gpt-4o';
            $aiProvider = new \Joomla\Plugin\System\AiAssistant\AI\OpenAIProvider($apiKey, $model);
        }

        // Initialize other components
        $actionRegistry = new \Joomla\Plugin\System\AiAssistant\Actions\ActionRegistry($params);
        $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger($params['log_all_actions'] ?? true);

        return new \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator(
            $aiProvider,
            $actionRegistry,
            $logger,
            $params
        );
    }

    /**
     * Send JSON response
     *
     * @param   array  $data        Response data
     * @param   int    $statusCode  HTTP status code
     *
     * @return  void
     * @since   1.0.0
     */
    private function sendJsonResponse(array $data, int $statusCode = 200): void
    {
        $app = $this->getApplication();
        
        $app->setHeader('Content-Type', 'application/json', true);
        $app->setHeader('Status', $statusCode, true);
        
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        $app->close();
    }

    /**
     * Check rate limiting
     *
     * @return  boolean  True if within rate limit
     * @since   1.0.0
     */
    private function checkRateLimit(): bool
    {
        $app = $this->getApplication();
        $session = $app->getSession();
        $now = time();
        
        // Get rate limit settings from params
        $maxRequests = (int) $this->params->get('rate_limit_requests', 10);
        $timeWindow = (int) $this->params->get('rate_limit_window', 60); // seconds
        
        // Get request history from session
        $requestHistory = $session->get('aiassistant_rate_limit', []);
        
        // Clean old requests outside the time window
        $requestHistory = array_filter($requestHistory, function($timestamp) use ($now, $timeWindow) {
            return ($now - $timestamp) < $timeWindow;
        });
        
        // Check if limit exceeded
        if (count($requestHistory) >= $maxRequests) {
            return false;
        }
        
        // Add current request
        $requestHistory[] = $now;
        $session->set('aiassistant_rate_limit', $requestHistory);
        
        return true;
    }
}
