<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Extension;

use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\SubscriberInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

// Prevent direct access
defined('_JEXEC') or die;

/**
 * AI Assistant Plugin
 *
 * Provides an agentic AI system integrated into Joomla that can read and modify
 * site content intelligently based on natural language prompts.
 *
 * @since  1.0.0
 */
final class AiAssistant extends CMSPlugin implements SubscriberInterface
{
    use DatabaseAwareTrait;

    /**
     * Load the language file on instantiation.
     *
     * @var    boolean
     * @since  1.0.0
     */
    protected $autoloadLanguage = true;

    /**
     * Returns an array of events this subscriber will listen to.
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterInitialise'   => 'onAfterInitialise',
            'onBeforeCompileHead' => 'onBeforeCompileHead',
            'onGetIcons'          => 'onGetIcons',
        ];
    }

    /**
     * After initialise event handler
     * Handles console access and AJAX requests
     *
     * @return  void
     * @since   1.0.0
     */
    public function onAfterInitialise(): void
    {
        $app = $this->getApplication();
        
        if (!$app) {
            Log::add('Application object is null', Log::ERROR, 'aiassistant');
            return;
        }
        
        $input = $app->input;
        
        if (!$input) {
            Log::add('Input object is null', Log::ERROR, 'aiassistant');
            return;
        }

        // Check if this is an AJAX request for our plugin
        if ($input->get('option') === 'com_ajax' 
            && $input->get('plugin') === 'aiassistant'
            && $input->get('group') === 'system') {
            
            try {
                // Start output buffering as early as possible
                if ($input->get('format', 'json') !== 'raw') {
                    ob_start();
                }
                
                // Check permissions - user must be logged in and be a super user
                $user = $app->getIdentity();
                if (!$user || !$user->id || !$user->authorise('core.admin')) {
                    $this->sendJsonResponse(['error' => 'Unauthorized. Super User access required.'], 403);
                    return;
                }

                // Special diagnostic endpoint
                if ($input->get('task') === 'diagnostic') {
                    $this->showDiagnostic();
                    return;
                }

                // Handle different request types
                $format = $input->get('format', 'json');
                
                if ($format === 'raw') {
                    // Display console HTML
                    $this->displayConsole();
                } else {
                    // Handle AJAX API request
                    $this->handleAjaxRequest();
                }
            } catch (\Exception $e) {
                // Catch any errors and return JSON response
                Log::add('AI Assistant initialization error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString(), Log::ERROR, 'aiassistant');
                $this->sendJsonResponse(['error' => 'An error occurred: ' . $e->getMessage()], 500);
            } catch (\Throwable $e) {
                // Catch fatal errors too
                Log::add('AI Assistant fatal error: ' . $e->getMessage() . ' | Trace: ' . $e->getTraceAsString(), Log::ERROR, 'aiassistant');
                $this->sendJsonResponse(['error' => 'A fatal error occurred: ' . $e->getMessage()], 500);
            }
        }
    }

    /**
     * Before compile head event handler
     * Adds menu item to sidebar
     *
     * @return  void
     * @since   1.0.0
     */
    public function onBeforeCompileHead(): void
    {
        $app = $this->getApplication();

        if (!$app->isClient('administrator')) {
            return;
        }

        // Check if user has permission
        $user = $app->getIdentity();
        if (!$user || !$user->id || !$user->authorise('core.admin')) {
            return;
        }

        // Add menu item via JavaScript
        $doc = $app->getDocument();
        $wa = $doc->getWebAssetManager();
        
        $consoleUrl = Uri::root() . 'administrator/index.php?option=com_ajax&plugin=aiassistant&group=system&format=raw';
        $menuTitle = Text::_('PLG_SYSTEM_AIASSISTANT_MENU_TITLE');
        
        // Add menu item to sidebar
        $jsCode = "document.addEventListener('DOMContentLoaded', function() { " .
            "var sidebar = document.querySelector('.sidebar-nav ul.main-nav'); " .
            "if (!sidebar) return; " .
            "var li = document.createElement('li'); " .
            "li.className = 'item item-level-1'; " .
            "var a = document.createElement('a'); " .
            "a.href = '" . $consoleUrl . "'; " .
            "a.className = 'no-dropdown'; " .
            "a.target = '_blank'; " .
            "var iconSpan = document.createElement('span'); " .
            "iconSpan.className = 'icon-cube'; " .
            "iconSpan.setAttribute('aria-hidden', 'true'); " .
            "var textSpan = document.createElement('span'); " .
            "textSpan.className = 'sidebar-item-title'; " .
            "textSpan.textContent = '" . addslashes($menuTitle) . "'; " .
            "a.appendChild(iconSpan); " .
            "a.appendChild(textSpan); " .
            "li.appendChild(a); " .
            "sidebar.appendChild(li); " .
        "});";
        
        $wa->addInlineScript($jsCode);
    }

    /**
     * Add icon to administrator dashboard (Quick Icons)
     *
     * @param   string  $context  The context
     *
     * @return  array
     * @since   1.0.0
     */
    public function onGetIcons(string $context): array
    {
        if ($context !== 'mod_quickicon' || !$this->getApplication()->isClient('administrator')) {
            return [];
        }

        // Check if user has permission
        $user = $this->getApplication()->getIdentity();
        if (!$user || !$user->id || !$user->authorise('core.admin')) {
            return [];
        }

        $consoleUrl = Uri::root() . 'administrator/index.php?option=com_ajax&plugin=aiassistant&group=system&format=raw';

        return [[
            'link'   => $consoleUrl,
            'image'  => 'icon-cube',
            'icon'   => 'icon-cube',
            'text'   => Text::_('PLG_SYSTEM_AIASSISTANT_QUICKICON_TITLE'),
            'id'     => 'plg_system_aiassistant-quickicon',
            'group'  => 'MOD_QUICKICON_SYSTEM',
            'access' => ['core.admin', 'com_cpanel'],
            'target' => '_blank'
        ]];
    }

    /**
     * Show diagnostic information
     *
     * @return  void
     * @since   1.0.0
     */
    private function showDiagnostic(): void
    {
        $this->ensureParamsLoaded();
        $params = $this->params ? $this->params->toArray() : [];
        
        $diagnostic = [
            'plugin_enabled' => $this->params ? $this->params->get('enabled', false) : false,
            'ai_provider' => $params['ai_provider'] ?? 'not set',
            'openai_key_length' => strlen($params['openai_api_key'] ?? ''),
            'openai_key_start' => substr($params['openai_api_key'] ?? '', 0, 20) . '...',
            'openai_model' => $params['openai_model'] ?? 'not set',
            'anthropic_key_length' => strlen($params['anthropic_api_key'] ?? ''),
            'params_count' => count($params),
            'all_param_keys' => array_keys($params)
        ];
        
        header('Content-Type: text/html; charset=utf-8');
        echo '<h1>AI Assistant Diagnostic</h1>';
        echo '<pre>';
        echo json_encode($diagnostic, JSON_PRETTY_PRINT);
        echo '</pre>';
        
        $this->getApplication()->close();
    }

    /**
     * Display the console interface
     *
     * @return  void
     * @since   1.0.0
     */
    private function displayConsole(): void
    {
        try {
            $app = $this->getApplication();
            
            // Output HTML directly (com_ajax with format=raw doesn't have document object)
            header('Content-Type: text/html; charset=utf-8');
            
            // Include the console template
            $templatePath = JPATH_PLUGINS . '/system/aiassistant/tmpl/console.php';
            
            if (file_exists($templatePath)) {
                include $templatePath;
            } else {
                echo '<!DOCTYPE html>';
                echo '<html><head><meta charset="UTF-8"><title>AI Assistant Console - Error</title></head><body>';
                echo '<h1>AI Assistant Console</h1><p>Template not found at: ' . htmlspecialchars($templatePath) . '</p>';
                echo '</body></html>';
            }

            $app->close();
        } catch (\Exception $e) {
            Log::add('Console display error: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
            header('Content-Type: text/html; charset=utf-8');
            echo '<!DOCTYPE html>';
            echo '<html><head><meta charset="UTF-8"><title>AI Assistant Console - Error</title></head><body>';
            echo '<h1>AI Assistant Console Error</h1><p>Failed to load console. Please check error logs.</p>';
            echo '<p>Error: ' . htmlspecialchars($e->getMessage()) . '</p>';
            echo '</body></html>';
            exit;
        }
    }

    /**
     * Handle AJAX request
     *
     * @return  void
     * @since   1.0.0
     */
    private function handleAjaxRequest(): void
    {
        try {
            Log::add('[DEBUG] handleAjaxRequest() started', Log::DEBUG, 'aiassistant');
            
            $app = $this->getApplication();
            Log::add('[DEBUG] getApplication() returned: ' . ($app ? 'object' : 'null'), Log::DEBUG, 'aiassistant');
            
            if (!$app) {
                $this->sendJsonResponse(['error' => 'Application object is null'], 500);
                return;
            }
            
            $input = $app->input;
            Log::add('[DEBUG] app->input returned: ' . ($input ? 'object' : 'null'), Log::DEBUG, 'aiassistant');
            
            if (!$input) {
                $this->sendJsonResponse(['error' => 'Input object is null'], 500);
                return;
            }

            // Validate CSRF token for POST requests
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                // Get the token from POST data
                $jsonData = file_get_contents('php://input');
                $data = json_decode($jsonData, true);
                
                // Check if token exists in data
                $tokenFound = false;
                if (is_array($data)) {
                    foreach ($data as $key => $value) {
                        if (Session::checkToken($key)) {
                            $tokenFound = true;
                            break;
                        }
                    }
                }
                
                if (!$tokenFound) {
                    Log::add('CSRF token validation failed. Token keys in request: ' . json_encode(array_keys($data ?? [])), Log::ERROR, 'aiassistant');
                    Log::add('Expected token: ' . Session::getFormToken(), Log::INFO, 'aiassistant');
                    $this->sendJsonResponse(['error' => 'Invalid CSRF token. Please refresh the page.'], 403);
                    return;
                }
            }

            // Apply rate limiting to all requests (both GET and POST)
            if (!$this->checkRateLimit()) {
                $this->sendJsonResponse(['error' => 'Rate limit exceeded. Please wait before making another request.'], 429);
                return;
            }

            $task = $input->get('task', 'process');
            Log::add('[DEBUG] Task: ' . $task, Log::DEBUG, 'aiassistant');

            switch ($task) {
                case 'process':
                    Log::add('[DEBUG] About to call processPrompt()', Log::DEBUG, 'aiassistant');
                    $this->processPrompt();
                    Log::add('[DEBUG] processPrompt() completed', Log::DEBUG, 'aiassistant');
                    break;
                
                case 'execute':
                    $this->executeActions();
                    break;
                
                case 'history':
                    $this->getHistory();
                    break;
                
                case 'session':
                    $this->getSession();
                    break;
                
                default:
                    $this->sendJsonResponse(['error' => 'Unknown task'], 400);
            }

        } catch (\Exception $e) {
            // Log the full error with file and line
            Log::add(
                '[ERROR] AI Assistant error in handleAjaxRequest(): ' . $e->getMessage() . 
                ' | File: ' . $e->getFile() . 
                ' | Line: ' . $e->getLine() .
                ' | Trace: ' . $e->getTraceAsString(),
                Log::ERROR,
                'aiassistant'
            );
            // Return the actual error message to help debug
            $this->sendJsonResponse(['error' => $e->getMessage() . ' (File: ' . basename($e->getFile()) . ', Line: ' . $e->getLine() . ')'], 500);
        } catch (\Throwable $e) {
            Log::add(
                '[FATAL] AI Assistant fatal error in handleAjaxRequest(): ' . $e->getMessage() . 
                ' | File: ' . $e->getFile() . 
                ' | Line: ' . $e->getLine() .
                ' | Trace: ' . $e->getTraceAsString(),
                Log::ERROR,
                'aiassistant'
            );
            $this->sendJsonResponse(['error' => $e->getMessage() . ' (File: ' . basename($e->getFile()) . ', Line: ' . $e->getLine() . ')'], 500);
        }
    }

    /**
     * Process a user prompt
     *
     * @return  void
     * @since   1.0.0
     */
    private function processPrompt(): void
    {
        Log::add('[DEBUG] processPrompt() started', Log::DEBUG, 'aiassistant');
        
        $app = $this->getApplication();
        Log::add('[DEBUG] processPrompt: getApplication() = ' . ($app ? 'object' : 'null'), Log::DEBUG, 'aiassistant');
        
        $input = $app->input;
        Log::add('[DEBUG] processPrompt: app->input = ' . ($input ? 'object' : 'null'), Log::DEBUG, 'aiassistant');
        
        if (!$input) {
            $this->sendJsonResponse(['error' => 'Input object is null'], 500);
            return;
        }

        $jsonData = file_get_contents('php://input');
        Log::add('[DEBUG] processPrompt: jsonData length = ' . strlen($jsonData), Log::DEBUG, 'aiassistant');
        
        $data = json_decode($jsonData, true);
        Log::add('[DEBUG] processPrompt: data decoded, json_error = ' . json_last_error(), Log::DEBUG, 'aiassistant');

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->sendJsonResponse(['error' => 'Invalid JSON data'], 400);
            return;
        }

        $prompt = $data['prompt'] ?? '';
        $requireReview = $data['require_review'] ?? true;

        // Validate and sanitize prompt
        if (empty($prompt) || !is_string($prompt)) {
            $this->sendJsonResponse(['error' => 'Prompt is required and must be a string'], 400);
            return;
        }

        // Limit prompt length to prevent abuse
        if (strlen($prompt) > 10000) {
            $this->sendJsonResponse(['error' => 'Prompt is too long (max 10000 characters)'], 400);
            return;
        }

        try {
            Log::add('[DEBUG] About to call getOrchestrator()', Log::DEBUG, 'aiassistant');
            $orchestrator = $this->getOrchestrator();
            Log::add('[DEBUG] getOrchestrator() completed successfully, orchestrator = ' . ($orchestrator ? 'object' : 'null'), Log::DEBUG, 'aiassistant');
            
            Log::add('[DEBUG] About to call orchestrator->processPrompt()', Log::DEBUG, 'aiassistant');
            $result = $orchestrator->processPrompt($prompt, (bool) $requireReview);
            Log::add('[DEBUG] orchestrator->processPrompt() completed successfully', Log::DEBUG, 'aiassistant');
            
            $this->sendJsonResponse($result);

        } catch (\Exception $e) {
            Log::add('[ERROR] Exception in processPrompt: ' . $e->getMessage() . 
                ' | File: ' . $e->getFile() . 
                ' | Line: ' . $e->getLine() . 
                ' | Trace: ' . $e->getTraceAsString(), Log::ERROR, 'aiassistant');
            $this->sendJsonResponse(['error' => $e->getMessage() . ' (File: ' . basename($e->getFile()) . ', Line: ' . $e->getLine() . ')'], 500);
        } catch (\Throwable $e) {
            Log::add('[FATAL] Throwable in processPrompt: ' . $e->getMessage() . 
                ' | File: ' . $e->getFile() . 
                ' | Line: ' . $e->getLine() . 
                ' | Trace: ' . $e->getTraceAsString(), Log::ERROR, 'aiassistant');
            $this->sendJsonResponse(['error' => $e->getMessage() . ' (File: ' . basename($e->getFile()) . ', Line: ' . $e->getLine() . ')'], 500);
        }
    }

    /**
     * Execute reviewed actions
     *
     * @return  void
     * @since   1.0.0
     */
    private function executeActions(): void
    {
        $app = $this->getApplication();
        $input = $app->input;
        
        if (!$input) {
            $this->sendJsonResponse(['error' => 'Input object is null'], 500);
            return;
        }

        $jsonData = file_get_contents('php://input');
        $data = json_decode($jsonData, true);

        $sessionId = $data['session_id'] ?? '';
        $actionIds = $data['action_ids'] ?? [];

        if (empty($sessionId)) {
            $this->sendJsonResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $orchestrator = $this->getOrchestrator();
            $result = $orchestrator->executeReviewedActions($sessionId, $actionIds);
            
            $this->sendJsonResponse($result);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get session history
     *
     * @return  void
     * @since   1.0.0
     */
    private function getHistory(): void
    {
        $this->ensureParamsLoaded();
        
        try {
            $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger(
                $this->params ? $this->params->get('log_all_actions', true) : true
            );
            $sessions = $logger->getRecentSessions(20);
            
            $this->sendJsonResponse(['sessions' => $sessions]);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get specific session details
     *
     * @return  void
     * @since   1.0.0
     */
    private function getSession(): void
    {
        $this->ensureParamsLoaded();
        
        $app = $this->getApplication();
        $input = $app->input;
        
        if (!$input) {
            $this->sendJsonResponse(['error' => 'Input object is null'], 500);
            return;
        }

        $sessionId = $input->get('session_id', '', 'string');

        if (empty($sessionId)) {
            $this->sendJsonResponse(['error' => 'Session ID is required'], 400);
            return;
        }

        try {
            $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger(
                $this->params ? $this->params->get('log_all_actions', true) : true
            );
            $actions = $logger->getSessionActions($sessionId);
            
            $this->sendJsonResponse(['actions' => $actions]);

        } catch (\Exception $e) {
            $this->sendJsonResponse(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Get orchestrator instance
     *
     * @return  \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator
     * @since   1.0.0
     */
    /**
     * Ensure params are loaded from database
     *
     * @return  void
     * @since   1.0.0
     */
    private function ensureParamsLoaded(): void
    {
        Log::add('[DEBUG] ensureParamsLoaded() called, params is ' . ($this->params === null ? 'null' : get_class($this->params)), Log::DEBUG, 'aiassistant');
        if ($this->params === null) {
            try {
                Log::add('[DEBUG] Loading params from database', Log::DEBUG, 'aiassistant');
                
                // Use Factory::getDbo() for more reliable database access
                try {
                    $db = Factory::getDbo();
                    Log::add('[DEBUG] Factory::getDbo() returned: ' . ($db ? get_class($db) : 'null'), Log::DEBUG, 'aiassistant');
                } catch (\Throwable $e) {
                    Log::add('[FATAL] Error calling Factory::getDbo(): ' . $e->getMessage() . ' | File: ' . $e->getFile() . ' | Line: ' . $e->getLine(), Log::ERROR, 'aiassistant');
                    $this->params = new Registry();
                    return;
                }
                
                if (!$db) {
                    Log::add('[ERROR] Database is null, using empty params', Log::ERROR, 'aiassistant');
                    $this->params = new Registry();
                    return;
                }
                
                $query = $db->getQuery(true)
                    ->select($db->quoteName('params'))
                    ->from($db->quoteName('#__extensions'))
                    ->where($db->quoteName('element') . ' = ' . $db->quote('aiassistant'))
                    ->where($db->quoteName('folder') . ' = ' . $db->quote('system'));
                $db->setQuery($query);
                
                Log::add('[DEBUG] About to execute query', Log::DEBUG, 'aiassistant');
                $paramsJson = $db->loadResult();
                Log::add('[DEBUG] Query executed, paramsJson = ' . ($paramsJson ? strlen($paramsJson) . ' bytes' : 'null'), Log::DEBUG, 'aiassistant');
                
                if ($paramsJson !== null) {
                    $this->params = new Registry($paramsJson);
                    Log::add('[DEBUG] Params loaded from database successfully, Registry created', Log::DEBUG, 'aiassistant');
                } else {
                    Log::add('[WARN] No params found in database, using defaults', Log::WARNING, 'aiassistant');
                    $this->params = new Registry();
                }
            } catch (\Throwable $e) {
                Log::add('[FATAL] Failed to load params: ' . $e->getMessage() . 
                    ' | File: ' . $e->getFile() . 
                    ' | Line: ' . $e->getLine() . 
                    ' | Trace: ' . $e->getTraceAsString(), Log::ERROR, 'aiassistant');
                $this->params = new Registry();
            }
        } else {
            Log::add('[DEBUG] Params already loaded, skipping database query', Log::DEBUG, 'aiassistant');
        }
    }

    private function getOrchestrator(): \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator
    {
        Log::add('[DEBUG] Starting getOrchestrator()', Log::DEBUG, 'aiassistant');
        Log::add('[DEBUG] Before ensureParamsLoaded: $this->params = ' . ($this->params === null ? 'null' : get_class($this->params)), Log::DEBUG, 'aiassistant');
        
        $this->ensureParamsLoaded();
        
        Log::add('[DEBUG] After ensureParamsLoaded: $this->params = ' . ($this->params === null ? 'null' : get_class($this->params)), Log::DEBUG, 'aiassistant');
        
        // Ensure params are not null after loading
        if ($this->params === null) {
            Log::add('[ERROR] Params are still null after ensureParamsLoaded()', Log::ERROR, 'aiassistant');
            $this->params = new Registry();
        }
        
        Log::add('[DEBUG] About to call toArray() on params (class: ' . get_class($this->params) . ')', Log::DEBUG, 'aiassistant');
        
        try {
            $params = $this->params->toArray();
            Log::add('[DEBUG] toArray() completed successfully, params count: ' . count($params), Log::DEBUG, 'aiassistant');
        } catch (\Throwable $e) {
            Log::add('[FATAL] Error calling toArray(): ' . $e->getMessage() . ' | File: ' . $e->getFile() . ' | Line: ' . $e->getLine(), Log::ERROR, 'aiassistant');
            throw $e;
        }

        // Initialize AI provider
        $provider = $params['ai_provider'] ?? 'openai';
        
        if ($provider === 'anthropic') {
            $apiKey = $params['anthropic_api_key'] ?? '';
            $model = $params['anthropic_model'] ?? 'claude-sonnet-4.5-20250929';
            
            Log::add(
                "Using Anthropic: Model={$model}, Key Length=" . strlen($apiKey),
                Log::INFO,
                'aiassistant'
            );
            
            $aiProvider = new \Joomla\Plugin\System\AiAssistant\AI\AnthropicProvider($apiKey, $model);
        } else {
            $apiKey = $params['openai_api_key'] ?? '';
            $model = $params['openai_model'] ?? 'gpt-5';
            
            Log::add(
                "Using OpenAI: Model={$model}, Key Length=" . strlen($apiKey) . ", Key Start=" . substr($apiKey, 0, 15) . "...",
                Log::INFO,
                'aiassistant'
            );
            
            $aiProvider = new \Joomla\Plugin\System\AiAssistant\AI\OpenAIProvider($apiKey, $model);
        }

        // Initialize other components
        $actionRegistry = new \Joomla\Plugin\System\AiAssistant\Actions\ActionRegistry($params);
        $logger = new \Joomla\Plugin\System\AiAssistant\Logger\ActionLogger($params['log_all_actions'] ?? true);

        return new \Joomla\Plugin\System\AiAssistant\Agent\AgentOrchestrator(
            $aiProvider,
            $actionRegistry,
            $logger,
            $params
        );
    }

    /**
     * Send JSON response
     *
     * @param   array  $data        Response data
     * @param   int    $statusCode  HTTP status code
     *
     * @return  void
     * @since   1.0.0
     */
    private function sendJsonResponse(array $data, int $statusCode = 200): void
    {
        // Clean any output buffers to ensure pure JSON response
        while (ob_get_level() > 0) {
            ob_end_clean();
        }
        
        $app = $this->getApplication();
        
        // Set headers before any output
        header('Content-Type: application/json; charset=utf-8', true);
        http_response_code($statusCode);
        
        echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        $app->close();
    }

    /**
     * Check rate limiting
     *
     * @return  boolean  True if within rate limit
     * @since   1.0.0
     */
    private function checkRateLimit(): bool
    {
        $this->ensureParamsLoaded();
        
        $app = $this->getApplication();
        
        if (!$app) {
            Log::add('Application object is null in checkRateLimit', Log::ERROR, 'aiassistant');
            return false;
        }
        
        $session = $app->getSession();
        
        if (!$session) {
            Log::add('Session object is null in checkRateLimit', Log::ERROR, 'aiassistant');
            return false;
        }
        
        $now = time();
        
        // Get rate limit settings from params
        $maxRequests = (int) ($this->params ? $this->params->get('rate_limit_requests', 10) : 10);
        $timeWindow = (int) ($this->params ? $this->params->get('rate_limit_window', 60) : 60); // seconds
        
        // Get request history from session
        $requestHistory = $session->get('aiassistant_rate_limit', []);
        
        // Clean old requests outside the time window
        $requestHistory = array_filter($requestHistory, function($timestamp) use ($now, $timeWindow) {
            return ($now - $timestamp) < $timeWindow;
        });
        
        // Check if limit exceeded
        if (count($requestHistory) >= $maxRequests) {
            return false;
        }
        
        // Add current request
        $requestHistory[] = $now;
        $session->set('aiassistant_rate_limit', $requestHistory);
        
        return true;
    }
}
