<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Installer\InstallerAdapter;
use Joomla\CMS\Installer\InstallerScriptInterface;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseInterface;
use Joomla\DI\Container;
use Joomla\DI\ServiceProviderInterface;

/**
 * Installation script for AI Assistant plugin
 *
 * @since  1.0.0
 */
return new class implements InstallerScriptInterface
{
    /**
     * Minimum Joomla version required
     */
    private string $minimumJoomla = '5.0.0';

    /**
     * Minimum PHP version required
     */
    private string $minimumPhp = '8.2.0';

    /**
     * Function called before installation
     *
     * @param   InstallerAdapter  $adapter  The adapter instance
     *
     * @return  boolean  True on success
     */
    public function install(InstallerAdapter $adapter): bool
    {
        echo '<h2>AI Assistant Plugin Installed Successfully!</h2>';
        echo '<p>Next steps:</p>';
        echo '<ol>';
        echo '<li>Enable the plugin in System → Plugins</li>';
        echo '<li>Configure your OpenAI or Anthropic API key</li>';
        echo '<li>Set action permissions</li>';
        echo '<li>Access the console at: administrator/index.php?option=com_ajax&plugin=aiassistant&group=system&format=raw</li>';
        echo '</ol>';
        
        return true;
    }

    /**
     * Function called after installation
     *
     * @param   InstallerAdapter  $adapter  The adapter instance
     *
     * @return  boolean  True on success
     */
    public function update(InstallerAdapter $adapter): bool
    {
        echo '<h2>AI Assistant Plugin Updated Successfully!</h2>';
        return true;
    }

    /**
     * Function called before uninstallation
     *
     * @param   InstallerAdapter  $adapter  The adapter instance
     *
     * @return  boolean  True on success
     */
    public function uninstall(InstallerAdapter $adapter): bool
    {
        echo '<h2>AI Assistant Plugin Uninstalled</h2>';
        echo '<p>Note: Database tables have been preserved. To remove them, run:</p>';
        echo '<pre>DROP TABLE IF EXISTS `#__aiassistant_sessions`;<br>';
        echo 'DROP TABLE IF EXISTS `#__aiassistant_actions`;</pre>';
        
        return true;
    }

    /**
     * Function called before any installation action
     *
     * @param   string            $type     Type of installation
     * @param   InstallerAdapter  $adapter  The adapter instance
     *
     * @return  boolean  True on success
     */
    public function preflight(string $type, InstallerAdapter $adapter): bool
    {
        // Check Joomla version
        if (version_compare(JVERSION, $this->minimumJoomla, '<')) {
            Factory::getApplication()->enqueueMessage(
                sprintf('AI Assistant requires Joomla %s or higher. You have %s', $this->minimumJoomla, JVERSION),
                'error'
            );
            return false;
        }

        // Check PHP version
        if (version_compare(PHP_VERSION, $this->minimumPhp, '<')) {
            Factory::getApplication()->enqueueMessage(
                sprintf('AI Assistant requires PHP %s or higher. You have %s', $this->minimumPhp, PHP_VERSION),
                'error'
            );
            return false;
        }

        return true;
    }

    /**
     * Function called after any installation action
     *
     * @param   string            $type     Type of installation
     * @param   InstallerAdapter  $adapter  The adapter instance
     *
     * @return  boolean  True on success
     */
    public function postflight(string $type, InstallerAdapter $adapter): bool
    {
        // Create database tables if they don't exist
        $this->createDatabaseTables();

        return true;
    }

    /**
     * Create database tables
     *
     * @return  void
     */
    private function createDatabaseTables(): void
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);

        try {
            // Create sessions table
            $query = "CREATE TABLE IF NOT EXISTS `#__aiassistant_sessions` (
                `id` VARCHAR(50) NOT NULL,
                `user_id` INT NOT NULL,
                `prompt` TEXT NOT NULL,
                `status` VARCHAR(50) NOT NULL,
                `result` MEDIUMTEXT NULL,
                `started_at` DATETIME NOT NULL,
                `ended_at` DATETIME NULL,
                PRIMARY KEY (`id`),
                INDEX `idx_user_started` (`user_id`, `started_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

            $db->setQuery($query);
            $db->execute();

            // Create actions table
            $query = "CREATE TABLE IF NOT EXISTS `#__aiassistant_actions` (
                `id` INT AUTO_INCREMENT,
                `session_id` VARCHAR(50) NOT NULL,
                `action_type` VARCHAR(100) NOT NULL,
                `parameters` TEXT NULL,
                `result` TEXT NULL,
                `status` VARCHAR(50) NOT NULL,
                `error` TEXT NULL,
                `executed_at` DATETIME NOT NULL,
                PRIMARY KEY (`id`),
                INDEX `idx_session` (`session_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

            $db->setQuery($query);
            $db->execute();

            Factory::getApplication()->enqueueMessage(
                'Database tables created successfully',
                'message'
            );

        } catch (\Exception $e) {
            Factory::getApplication()->enqueueMessage(
                'Warning: Could not create database tables. They will be created on first use. Error: ' . $e->getMessage(),
                'warning'
            );
        }
    }
};

