const { Pool } = require('pg');

// Database configuration
// For Cloud SQL, use Unix socket connection when INSTANCE_UNIX_SOCKET is set
const dbConfig = process.env.INSTANCE_UNIX_SOCKET
  ? {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.INSTANCE_UNIX_SOCKET,
    }
  : {
      user: process.env.DB_USER || 'postgres',
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME || 'photowebsite',
      host: process.env.DB_HOST || 'localhost',
      port: parseInt(process.env.DB_PORT || '5432', 10),
    };

const pool = new Pool(dbConfig);

// Initialize database schema
async function initDatabase() {
  const client = await pool.connect();
  try {
    await client.query(`
      CREATE TABLE IF NOT EXISTS album_groups (
        id SERIAL PRIMARY KEY,
        name TEXT NOT NULL,
        code TEXT NOT NULL UNIQUE,
        description TEXT,
        created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
      );

      CREATE TABLE IF NOT EXISTS albums (
        id SERIAL PRIMARY KEY,
        title TEXT NOT NULL,
        code TEXT NOT NULL UNIQUE,
        created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
        cover_photo_id INTEGER,
        group_id INTEGER REFERENCES album_groups(id) ON DELETE SET NULL,
        is_public INTEGER NOT NULL DEFAULT 0
      );
      
      CREATE TABLE IF NOT EXISTS photos (
        id SERIAL PRIMARY KEY,
        album_id INTEGER NOT NULL REFERENCES albums(id) ON DELETE CASCADE,
        filename TEXT NOT NULL,
        original_name TEXT,
        mime_type TEXT,
        size_bytes BIGINT,
        created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
      );

      CREATE INDEX IF NOT EXISTS idx_albums_code ON albums(code);
      CREATE INDEX IF NOT EXISTS idx_albums_group_id ON albums(group_id);
      CREATE INDEX IF NOT EXISTS idx_albums_is_public ON albums(is_public);
      CREATE INDEX IF NOT EXISTS idx_photos_album_id ON photos(album_id);
      CREATE INDEX IF NOT EXISTS idx_album_groups_code ON album_groups(code);
    `);
    console.log('✓ Database schema initialized');
  } finally {
    client.release();
  }
}

// Initialize on startup
initDatabase().catch(err => {
  console.error('Failed to initialize database:', err);
  process.exit(1);
});

async function createAlbum(title, code) {
  const result = await pool.query(
    'INSERT INTO albums (title, code) VALUES ($1, $2) RETURNING id',
    [title, code]
  );
  return result.rows[0].id;
}

async function listAlbums() {
  const result = await pool.query(
    'SELECT id, title, code, created_at FROM albums ORDER BY created_at DESC'
  );
  return result.rows;
}

async function getAlbumById(albumId) {
  const result = await pool.query('SELECT * FROM albums WHERE id = $1', [albumId]);
  return result.rows[0] || null;
}

async function getAlbumByCode(code) {
  const result = await pool.query('SELECT * FROM albums WHERE code = $1', [code]);
  return result.rows[0] || null;
}

async function updateAlbumTitle(albumId, title) {
  await pool.query('UPDATE albums SET title = $1 WHERE id = $2', [title, albumId]);
}

async function addPhoto(albumId, filename, originalName, mimeType, sizeBytes) {
  const result = await pool.query(
    'INSERT INTO photos (album_id, filename, original_name, mime_type, size_bytes) VALUES ($1, $2, $3, $4, $5) RETURNING id',
    [albumId, filename, originalName, mimeType, sizeBytes]
  );
  return result.rows[0].id;
}

async function getPhotosByAlbum(albumId) {
  const result = await pool.query(
    'SELECT * FROM photos WHERE album_id = $1 ORDER BY created_at DESC',
    [albumId]
  );
  return result.rows;
}

// Album Group functions
async function createGroup(name, code, description = null) {
  const result = await pool.query(
    'INSERT INTO album_groups (name, code, description) VALUES ($1, $2, $3) RETURNING id',
    [name, code, description]
  );
  return result.rows[0].id;
}

async function listGroups() {
  const result = await pool.query(
    'SELECT id, name, code, description, created_at FROM album_groups ORDER BY created_at DESC'
  );
  return result.rows;
}

async function getGroupById(groupId) {
  const result = await pool.query('SELECT * FROM album_groups WHERE id = $1', [groupId]);
  return result.rows[0] || null;
}

async function getGroupByCode(code) {
  const result = await pool.query('SELECT * FROM album_groups WHERE code = $1', [code]);
  return result.rows[0] || null;
}

async function getAlbumsByGroup(groupId) {
  const result = await pool.query(
    'SELECT * FROM albums WHERE group_id = $1 ORDER BY created_at DESC',
    [groupId]
  );
  return result.rows;
}

async function assignAlbumToGroup(albumId, groupId) {
  await pool.query('UPDATE albums SET group_id = $1 WHERE id = $2', [groupId, albumId]);
}

async function removeAlbumFromGroup(albumId) {
  await pool.query('UPDATE albums SET group_id = NULL WHERE id = $1', [albumId]);
}

async function getGroupForAlbum(albumId) {
  const result = await pool.query(
    'SELECT ag.* FROM album_groups ag JOIN albums a ON a.group_id = ag.id WHERE a.id = $1',
    [albumId]
  );
  return result.rows[0] || null;
}

async function listPublicAlbums() {
  const result = await pool.query(`
    SELECT a.*, 
           (SELECT p.filename 
            FROM photos p 
            WHERE p.album_id = a.id 
            ORDER BY p.created_at ASC 
            LIMIT 1) as cover_filename,
           (SELECT COUNT(*) FROM photos p WHERE p.album_id = a.id) as photo_count
    FROM albums a 
    WHERE a.is_public = 1 
    ORDER BY a.created_at DESC
  `);
  return result.rows;
}

async function toggleAlbumPublic(albumId, isPublic) {
  await pool.query('UPDATE albums SET is_public = $1 WHERE id = $2', [isPublic ? 1 : 0, albumId]);
}

module.exports = {
  createAlbum,
  listAlbums,
  getAlbumById,
  getAlbumByCode,
  updateAlbumTitle,
  addPhoto,
  getPhotosByAlbum,
  createGroup,
  listGroups,
  getGroupById,
  getGroupByCode,
  getAlbumsByGroup,
  assignAlbumToGroup,
  removeAlbumFromGroup,
  getGroupForAlbum,
  listPublicAlbums,
  toggleAlbumPublic,
  pool,
};

