const path = require('path');
const fs = require('fs');
const fsPromises = require('fs').promises;
require('dotenv').config();
const express = require('express');
const cookieParser = require('cookie-parser');
const multer = require('multer');
const { customAlphabet } = require('nanoid');
const sharp = require('sharp');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const db = require('./db');
const storage = require('./storage');

const app = express();
const PORT = process.env.PORT || 3000;

// Admin password - warn if not set in production
const ADMIN_PASSWORD = process.env.ADMIN_PASSWORD || 'changeme';
if (!process.env.ADMIN_PASSWORD) {
  console.warn('[SECURITY WARNING] ADMIN_PASSWORD environment variable not set. Using default password.');
  console.warn('[SECURITY WARNING] Please set ADMIN_PASSWORD in your environment for production!');
}

// Site configuration from environment variables
const AGE_VERIFICATION_ENABLED = process.env.AGE_VERIFICATION_ENABLED !== 'false';
const PRIMARY_COLOR = process.env.PRIMARY_COLOR || '#a6ff00';
const LOGO_URL = process.env.LOGO_URL || '/logo.png';
const SITE_NAME = process.env.SITE_NAME || 'Lucky Pup Photography';

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, '..', 'views'));

// Security headers
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
      fontSrc: ["'self'", "https://fonts.gstatic.com"],
      imgSrc: ["'self'", "data:", "https://storage.googleapis.com", "https://storage.cloud.google.com"],
      scriptSrc: ["'self'", "'unsafe-inline'"],
      connectSrc: ["'self'"]
    }
  },
  crossOriginEmbedderPolicy: false
}));

// Rate limiters
const generalLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // 100 requests per window
  message: 'Too many requests, please try again later.',
  standardHeaders: true,
  legacyHeaders: false
});

const authLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 5, // 5 attempts per window
  message: 'Too many authentication attempts, please try again later.',
  standardHeaders: true,
  legacyHeaders: false
});

const uploadLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 50, // 50 uploads per hour
  message: 'Too many uploads, please try again later.',
  standardHeaders: true,
  legacyHeaders: false
});

app.use(generalLimiter);
app.use(express.urlencoded({ extended: true, limit: '1mb' }));
app.use(express.json({ limit: '1mb' }));
app.use(cookieParser());
app.use(express.static(path.join(__dirname, '..', 'public')));
app.use('/uploads', express.static(path.join(__dirname, '..', 'uploads')));

// Middleware to inject site config into all views
app.use(async (req, res, next) => {
  try {
    const config = await db.getSiteConfig();
    res.locals.siteConfig = config;
    // Log config on first load only
    if (!global._configLogged) {
      console.log('[CONFIG] Loaded site config:', config);
      global._configLogged = true;
    }
    next();
  } catch (err) {
    console.error('Error loading site config:', err);
    // Fallback config
    res.locals.siteConfig = {
      logoUrl: process.env.LOGO_URL || '/logo.png',
      primaryColor: process.env.PRIMARY_COLOR || '#a6ff00',
      siteName: process.env.SITE_NAME || 'Lucky Pup Photography',
      hotChocolateDefaultUrl: 'https://square.link/u/NkgDiQCk',
      hotChocolateText: 'Like these pics? Buy me a Hot Chocolate',
      contactEmail: '',
      socialLinks: { facebook: '', twitter: '', instagram: '', telegram: '', bluesky: '' }
    };
    next();
  }
});

// Basic Auth for admin routes
function requireAdmin(req, res, next) {
  const header = req.headers['authorization'] || '';
  const token = header.split(' ')[1] || '';
  const expected = Buffer.from(`admin:${ADMIN_PASSWORD}`).toString('base64');
  if (!token || token !== expected) {
    console.warn(`[SECURITY] Failed admin login attempt from ${req.ip} at ${new Date().toISOString()}`);
    res.set('WWW-Authenticate', 'Basic realm="Lucky Pup Admin"');
    return res.status(401).send('Authentication required');
  }
  console.log(`[SECURITY] Successful admin login from ${req.ip} at ${new Date().toISOString()}`);
  next();
}

// Age verification middleware for public routes
function requireAgeVerification(req, res, next) {
  // Skip age verification if disabled
  if (!AGE_VERIFICATION_ENABLED) {
    return next();
  }
  
  // Check if age verification cookie exists
  if (req.cookies.age_verified === 'true') {
    return next();
  }
  
  // If not verified, pass a flag to show the age gate
  res.locals.showAgeGate = true;
  res.locals.originalUrl = req.originalUrl;
  next();
}

// Nanoid code generator (A-Z 0-9, 8 chars)
const alphabet = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
const nanoShort = customAlphabet(alphabet, 8);

// Input validation helpers
function validateTitle(title) {
  return title && typeof title === 'string' && title.trim().length > 0 && title.length <= 200;
}

function validateDescription(description) {
  return !description || (typeof description === 'string' && description.length <= 1000);
}

function validateUrl(url) {
  if (!url) return true; // Optional
  try {
    new URL(url);
    return url.length <= 500;
  } catch {
    return false;
  }
}

// Generate thumbnail for an image
async function generateThumbnail(albumId, filename) {
  try {
    await storage.generateAndUploadThumbnail(null, albumId, filename);
    return true;
  } catch (error) {
    console.error('Error generating thumbnail:', error);
    return false;
  }
}

// Make storage helper and site config available to all views
app.use((req, res, next) => {
  res.locals.storage = storage;
  res.locals.siteConfig = {
    primaryColor: PRIMARY_COLOR,
    logoUrl: LOGO_URL,
    siteName: SITE_NAME,
    ageVerificationEnabled: AGE_VERIFICATION_ENABLED
  };
  next();
});

// Routes - Public
app.get('/', requireAgeVerification, async (req, res) => {
  const publicAlbums = await db.listPublicAlbums();
  const homePageAlbum = await db.getHomePageAlbum();
  let homePagePhotos = [];
  
  if (homePageAlbum) {
    homePagePhotos = await db.getPhotosByAlbum(homePageAlbum.id);
  }
  
  res.render('index', { 
    pageTitle: 'Access Your Album', 
    publicAlbums, 
    homePageAlbum, 
    homePagePhotos 
  });
});

// Age verification endpoint
app.post('/verify-age', authLimiter, (req, res) => {
  const day = parseInt(req.body.day, 10);
  const month = parseInt(req.body.month, 10);
  const year = parseInt(req.body.year, 10);
  
  // Validate inputs
  if (!day || !month || !year || day < 1 || day > 31 || month < 1 || month > 12 || year < 1900) {
    return res.redirect(req.body.return_to || '/');
  }
  
  // Calculate age
  const birthDate = new Date(year, month - 1, day);
  const today = new Date();
  let age = today.getFullYear() - birthDate.getFullYear();
  const monthDiff = today.getMonth() - birthDate.getMonth();
  
  // Adjust age if birthday hasn't occurred yet this year
  if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthDate.getDate())) {
    age--;
  }
  
  // Check if user is 18 or older
  if (age >= 18) {
    // Set cookie for 30 days
    res.cookie('age_verified', 'true', {
      maxAge: 30 * 24 * 60 * 60 * 1000,
      httpOnly: true,
      sameSite: 'strict'
    });
    
    // Redirect to where they came from or homepage
    const returnTo = req.body.return_to || '/';
    return res.redirect(returnTo);
  } else {
    // If they're under 18, redirect them away
    return res.redirect('https://www.bbc.co.uk/cbbc');
  }
});

app.post('/access', requireAgeVerification, async (req, res) => {
  const code = String(req.body.code || '').trim().toUpperCase();
  if (!code) return res.redirect('/');
  
  // Check if it's a group code first
  const group = await db.getGroupByCode(code);
  if (group) {
    return res.redirect(`/g/${encodeURIComponent(code)}`);
  }
  
  // Otherwise, try album
  return res.redirect(`/a/${encodeURIComponent(code)}`);
});

app.get('/a/:code', requireAgeVerification, async (req, res) => {
  const code = String(req.params.code || '').toUpperCase();
  const album = await db.getAlbumByCode(code);
  if (!album) {
    return res.status(404).render('not-found', { pageTitle: 'Album Not Found' });
  }
  const photos = await db.getPhotosByAlbum(album.id);
  res.render('album', { pageTitle: album.title, album, photos });
});

app.get('/g/:code', requireAgeVerification, async (req, res) => {
  const code = String(req.params.code || '').toUpperCase();
  const group = await db.getGroupByCode(code);
  if (!group) {
    return res.status(404).render('not-found', { pageTitle: 'Group Not Found' });
  }
  const albums = await db.getAlbumsByGroup(group.id);
  res.render('group', { pageTitle: group.name, group, albums });
});

// Serve or generate thumbnail on-demand
app.get('/thumb/:albumId/:filename', async (req, res) => {
  // Sanitize inputs to prevent path traversal
  const albumId = String(req.params.albumId).replace(/[^0-9]/g, '');
  const filename = path.basename(req.params.filename);
  
  if (!albumId || !filename) {
    return res.status(400).send('Invalid request');
  }
  
  if (storage.USE_CLOUD_STORAGE) {
    // For Cloud Storage, redirect to the public URL
    const thumbUrl = storage.getPublicUrl(albumId, filename, true);
    return res.redirect(thumbUrl);
  }
  
  // Local development - serve from filesystem
  const thumbPath = path.join(__dirname, '..', 'uploads', albumId, 'thumbs', filename);
  
  // If thumbnail exists, serve it
  if (fs.existsSync(thumbPath)) {
    return res.sendFile(thumbPath);
  }
  
  // Otherwise, generate it on-demand
  const sourcePath = path.join(__dirname, '..', 'uploads', albumId, filename);
  
  // Check if source exists
  if (!fs.existsSync(sourcePath)) {
    return res.status(404).send('Image not found');
  }
  
  try {
    await generateThumbnail(albumId, filename);
    res.sendFile(thumbPath);
  } catch (error) {
    console.error('Error serving thumbnail:', error);
    // Fallback to original image
    res.sendFile(sourcePath);
  }
});

// Multer storage configured per album
const multerStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    const albumId = req.params.id;
    const albumDir = path.join(__dirname, '..', 'uploads', String(albumId));
    fs.mkdirSync(albumDir, { recursive: true });
    cb(null, albumDir);
  },
  filename: function (req, file, cb) {
    const timestamp = Date.now();
    const safeOriginal = path.basename(file.originalname).replace(/[^a-zA-Z0-9._-]/g, '_');
    cb(null, `${timestamp}-${safeOriginal}`);
  }
});

function imageOnlyFilter(req, file, cb) {
  if ((file.mimetype || '').startsWith('image/')) return cb(null, true);
  cb(new Error('Only image uploads are allowed'));
}

const upload = multer({ 
  storage: multerStorage, 
  fileFilter: imageOnlyFilter,
  limits: {
    fileSize: 50 * 1024 * 1024, // 50MB per file
    files: 500 // Allow up to 500 files at once
  }
});

// Admin routes
app.get('/admin', requireAdmin, async (req, res) => {
  const albums = await db.listAlbums();
  const groups = await db.listGroups();
  res.render('admin/index', { pageTitle: 'Admin', albums, groups });
});

app.get('/admin/settings', requireAdmin, async (req, res) => {
  const config = await db.getSiteConfig();
  res.render('admin/settings', { pageTitle: 'Site Settings', config });
});

app.post('/admin/settings', requireAdmin, async (req, res) => {
  const updates = {
    primaryColor: String(req.body.primaryColor || '').trim(),
    siteName: String(req.body.siteName || '').trim(),
    hotChocolateDefaultUrl: String(req.body.hotChocolateDefaultUrl || '').trim(),
    hotChocolateText: String(req.body.hotChocolateText || '').trim(),
    contactEmail: String(req.body.contactEmail || '').trim(),
    socialLinks: {
      facebook: String(req.body.facebook || '').trim(),
      twitter: String(req.body.twitter || '').trim(),
      instagram: String(req.body.instagram || '').trim(),
      telegram: String(req.body.telegram || '').trim(),
      bluesky: String(req.body.bluesky || '').trim()
    }
  };
  
  console.log('[CONFIG] Saving settings:', updates);
  
  // Validate inputs
  if (updates.siteName && !validateTitle(updates.siteName)) {
    return res.status(400).send('Invalid site name');
  }
  
  if (updates.hotChocolateText && !validateDescription(updates.hotChocolateText)) {
    return res.status(400).send('Invalid hot chocolate text');
  }
  
  if (updates.hotChocolateDefaultUrl && !validateUrl(updates.hotChocolateDefaultUrl)) {
    return res.status(400).send('Invalid hot chocolate URL');
  }
  
  await db.updateSiteConfig(updates);
  console.log('[CONFIG] Settings saved successfully');
  res.redirect('/admin/settings');
});

app.post('/admin/settings/logo', requireAdmin, upload.single('logo'), async (req, res) => {
  if (!req.file) {
    return res.status(400).send('No logo file uploaded');
  }
  
  const allowedTypes = ['image/png', 'image/jpeg', 'image/jpg', 'image/svg+xml'];
  if (!allowedTypes.includes(req.file.mimetype)) {
    // Delete uploaded file
    fsPromises.unlink(req.file.path).catch(() => {});
    return res.status(400).send('Invalid file type. Please upload PNG, JPG, or SVG.');
  }
  
  // Move logo to public directory
  const ext = path.extname(req.file.originalname);
  const logoFilename = `logo-custom${ext}`;
  const logoPath = path.join(__dirname, '..', 'public', logoFilename);
  
  await fsPromises.rename(req.file.path, logoPath);
  
  // Update config with new logo URL
  await db.updateSiteConfig({ logoUrl: `/${logoFilename}` });
  
  res.redirect('/admin/settings');
});

app.post('/admin/albums', requireAdmin, async (req, res) => {
  const title = String(req.body.title || '').trim();
  
  if (!validateTitle(title)) {
    return res.status(400).send('Invalid album title (max 200 characters)');
  }
  
  const code = nanoShort();
  const id = await db.createAlbum(title, code);
  return res.redirect(`/admin/albums/${id}`);
});

app.get('/admin/albums/:id', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  const photos = await db.getPhotosByAlbum(id);
  const groups = await db.listGroups();
  const albumGroup = await db.getGroupForAlbum(id);
  res.render('admin/album', { pageTitle: album.title, album, photos, groups, albumGroup });
});

app.post('/admin/albums/:id/upload', requireAdmin, uploadLimiter, upload.array('photos', 500), async (req, res) => {
  const id = Number(req.params.id);
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  // Add photos to DB and upload to Cloud Storage if enabled
  for (const file of req.files || []) {
    if (storage.USE_CLOUD_STORAGE) {
      // Upload to Cloud Storage
      await storage.uploadFile(file.path, `${id}/${file.filename}`);
      // Delete local temp file
      fs.unlinkSync(file.path);
    }
    
    await db.addPhoto(id, file.filename, file.originalname, file.mimetype, file.size);
    
    // Generate thumbnail asynchronously (don't wait)
    generateThumbnail(id, file.filename).catch(err => 
      console.error(`Failed to generate thumbnail for ${file.filename}:`, err)
    );
  }
  
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/group', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const groupId = req.body.group_id ? Number(req.body.group_id) : null;
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  if (groupId) {
    await db.assignAlbumToGroup(id, groupId);
  } else {
    await db.removeAlbumFromGroup(id);
  }
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/rename', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const title = String(req.body.title || '').trim();
  
  if (!validateTitle(title)) {
    return res.status(400).send('Invalid album title (max 200 characters)');
  }
  
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  if (title) {
    await db.updateAlbumTitle(id, title);
  }
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/description', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const description = String(req.body.description || '').trim();
  
  if (!validateDescription(description)) {
    return res.status(400).send('Invalid description (max 1000 characters)');
  }
  
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  await db.updateAlbumDescription(id, description || null);
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/public', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const isPublic = req.body.is_public === '1';
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  await db.toggleAlbumPublic(id, isPublic);
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/hot-chocolate', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const enabled = req.body.enabled === '1';
  let url = String(req.body.url || '').trim();
  
  // Default to the configured default URL if no URL provided
  if (enabled && !url) {
    const config = await db.getSiteConfig();
    url = config.hotChocolateDefaultUrl;
  }
  
  // Validate URL if provided
  if (url && !validateUrl(url)) {
    return res.status(400).send('Invalid URL (max 500 characters)');
  }
  
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  await db.updateHotChocolateSettings(id, enabled, url || null);
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/:id/home-page', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  await db.setHomePageAlbum(id);
  res.redirect(`/admin/albums/${id}`);
});

app.post('/admin/albums/reorder', requireAdmin, async (req, res) => {
  try {
    const { albumIds } = req.body;
    
    console.log('Album reorder request - Full array:', JSON.stringify(albumIds));
    console.log('Album reorder request - First 3 IDs:', albumIds.slice(0, 3));
    
    if (!albumIds || !Array.isArray(albumIds)) {
      console.error('Invalid album IDs:', albumIds);
      return res.status(400).json({ error: 'Invalid album IDs' });
    }
    
    await db.updateAlbumSortOrder(albumIds.map(id => Number(id)));
    console.log('Album order updated successfully');
    res.json({ success: true });
  } catch (error) {
    console.error('Error reordering albums:', error);
    res.status(500).json({ error: 'Failed to reorder albums', details: error.message });
  }
});

app.post('/admin/albums/:id/delete', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const album = await db.getAlbumById(id);
  if (!album) return res.redirect('/admin');
  
  // TODO: Delete files from Cloud Storage
  await db.deleteAlbum(id);
  res.redirect('/admin');
});

// Photo management routes
app.post('/admin/albums/:albumId/photos/reorder', requireAdmin, async (req, res) => {
  try {
    const albumId = Number(req.params.albumId);
    const { photoIds } = req.body;
    
    console.log('Reorder request:', { albumId, photoIds });
    
    if (!photoIds || !Array.isArray(photoIds)) {
      console.error('Invalid photo IDs:', photoIds);
      return res.status(400).json({ error: 'Invalid photo IDs' });
    }
    
    await db.updatePhotoSortOrder(photoIds.map(id => Number(id)));
    console.log('Photo order updated successfully');
    res.json({ success: true });
  } catch (error) {
    console.error('Error reordering photos:', error);
    res.status(500).json({ error: 'Failed to reorder photos', details: error.message });
  }
});

app.post('/admin/albums/:albumId/photos/:photoId/delete', requireAdmin, async (req, res) => {
  const albumId = Number(req.params.albumId);
  const photoId = Number(req.params.photoId);
  
  const photo = await db.getPhotoById(photoId);
  if (!photo || photo.album_id !== albumId) {
    return res.status(404).json({ error: 'Photo not found' });
  }
  
  // TODO: Delete file from Cloud Storage
  await db.deletePhoto(photoId);
  res.json({ success: true });
});

// Admin Group routes
app.post('/admin/groups', requireAdmin, async (req, res) => {
  const name = String(req.body.name || '').trim();
  const description = String(req.body.description || '').trim() || null;
  if (!name) return res.redirect('/admin');
  const code = nanoShort();
  const id = await db.createGroup(name, code, description);
  return res.redirect(`/admin/groups/${id}`);
});

app.get('/admin/groups/:id', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const group = await db.getGroupById(id);
  if (!group) return res.redirect('/admin');
  const albums = await db.getAlbumsByGroup(id);
  res.render('admin/group', { pageTitle: group.name, group, albums });
});

app.get('/admin/groups/:id/delete', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const group = await db.getGroupById(id);
  if (!group) return res.redirect('/admin');
  const albums = await db.getAlbumsByGroup(id);
  res.render('admin/delete-group', { pageTitle: 'Delete Group', group, albums });
});

app.post('/admin/groups/:id/delete', requireAdmin, async (req, res) => {
  const id = Number(req.params.id);
  const group = await db.getGroupById(id);
  if (!group) return res.redirect('/admin');
  
  const deleteAlbums = req.body.delete_albums === '1';
  await db.deleteGroup(id, deleteAlbums);
  res.redirect('/admin');
});

// 404 fallback
app.use((req, res) => {
  res.status(404).render('not-found', { pageTitle: 'Not Found' });
});

app.listen(PORT, () => {
  console.log(`Lucky Pup server running on http://localhost:${PORT}`);
});


