# Adding a New Project to Multi-Site Deployment

This guide shows how to add `nrwphotography` (or any other project) to your automated deployment workflow.

## Step 1: Enable Billing

⚠️ **Do this first before running any commands:**

1. Go to: https://console.cloud.google.com/billing/linkedaccount?project=nrwphotography
2. Link your billing account to the project
3. Confirm billing is enabled

## Step 2: Enable Required APIs

Run these commands after billing is enabled:

```bash
gcloud services enable run.googleapis.com \
  storage.googleapis.com \
  secretmanager.googleapis.com \
  cloudbuild.googleapis.com \
  --project=nrwphotography
```

## Step 3: Create Storage Bucket

```bash
# Create bucket in europe-west1 (free tier region)
gsutil mb -l europe-west1 gs://nrwphotography-photowebsite

# Make bucket publicly readable
gsutil iam ch allUsers:objectViewer gs://nrwphotography-photowebsite
```

## Step 4: Create Service Account for GitHub Actions

```bash
# Create service account
gcloud iam service-accounts create github-actions \
  --display-name="GitHub Actions Deployer" \
  --project=nrwphotography

# Grant necessary permissions
gcloud projects add-iam-policy-binding nrwphotography \
  --member="serviceAccount:github-actions@nrwphotography.iam.gserviceaccount.com" \
  --role="roles/run.admin"

gcloud projects add-iam-policy-binding nrwphotography \
  --member="serviceAccount:github-actions@nrwphotography.iam.gserviceaccount.com" \
  --role="roles/cloudbuild.builds.editor"

gcloud projects add-iam-policy-binding nrwphotography \
  --member="serviceAccount:github-actions@nrwphotography.iam.gserviceaccount.com" \
  --role="roles/storage.admin"

gcloud projects add-iam-policy-binding nrwphotography \
  --member="serviceAccount:github-actions@nrwphotography.iam.gserviceaccount.com" \
  --role="roles/iam.serviceAccountUser"

gcloud projects add-iam-policy-binding nrwphotography \
  --member="serviceAccount:github-actions@nrwphotography.iam.gserviceaccount.com" \
  --role="roles/viewer"

# Create and download key
gcloud iam service-accounts keys create nrw-github-actions-key.json \
  --iam-account=github-actions@nrwphotography.iam.gserviceaccount.com \
  --project=nrwphotography
```

## Step 5: Set Admin Password

```bash
# Create secret for admin password
echo -n "your-secure-password-here" | \
  gcloud secrets create photowebsite-admin-password \
  --data-file=- \
  --project=nrwphotography
```

## Step 6: Add Service Account Key to GitHub

1. Open `nrw-github-actions-key.json`
2. Copy the entire JSON content
3. Go to: https://github.com/thedoctor0124/photowebsite/settings/secrets/actions
4. Click "New repository secret"
5. Name: `GCP_SA_KEY_NRW`
6. Value: Paste the JSON content
7. Click "Add secret"
8. **Delete the key file:** `del nrw-github-actions-key.json`

## Step 7: Create Cloud Build YAML for NRW

Create `cloudbuild-nrw.yaml` with NRW-specific configuration:

```yaml
steps:
  # Build Docker image
  - name: 'gcr.io/cloud-builders/docker'
    args:
      - 'build'
      - '-t'
      - 'gcr.io/nrwphotography/photowebsite:${COMMIT_SHA}'
      - '-t'
      - 'gcr.io/nrwphotography/photowebsite:latest'
      - '.'

  # Push to Container Registry
  - name: 'gcr.io/cloud-builders/docker'
    args:
      - 'push'
      - '--all-tags'
      - 'gcr.io/nrwphotography/photowebsite'

  # Deploy to Cloud Run
  - name: 'gcr.io/google.com/cloudsdktool/cloud-sdk'
    entrypoint: gcloud
    args:
      - 'run'
      - 'deploy'
      - 'photowebsite'
      - '--image'
      - 'gcr.io/nrwphotography/photowebsite:${COMMIT_SHA}'
      - '--platform'
      - 'managed'
      - '--region'
      - '${_DEPLOY_REGION}'
      - '--allow-unauthenticated'
      - '--set-env-vars'
      - 'USE_JSON=true,USE_CLOUD_STORAGE=true,GCS_BUCKET_NAME=${_GCS_BUCKET_NAME},AGE_VERIFICATION_ENABLED=${_AGE_VERIFICATION_ENABLED},PRIMARY_COLOR=${_PRIMARY_COLOR},LOGO_URL=${_LOGO_URL},SITE_NAME=${_SITE_NAME}'
      - '--set-secrets'
      - 'ADMIN_PASSWORD=photowebsite-admin-password:latest'
      - '--memory'
      - '256Mi'
      - '--cpu'
      - '1'
      - '--max-instances'
      - '10'
      - '--min-instances'
      - '0'
      - '--timeout'
      - '300'

images:
  - 'gcr.io/nrwphotography/photowebsite:${COMMIT_SHA}'
  - 'gcr.io/nrwphotography/photowebsite:latest'

options:
  logging: CLOUD_LOGGING_ONLY

substitutions:
  _DEPLOY_REGION: 'europe-west1'
  _GCS_BUCKET_NAME: 'nrwphotography-photowebsite'
  _AGE_VERIFICATION_ENABLED: 'false'
  _PRIMARY_COLOR: '#3366ff'
  _LOGO_URL: '/nrw-web.png'
  _SITE_NAME: 'NRW Web'
```

## Step 8: Update GitHub Workflow

The workflow has already been updated to use a matrix strategy that deploys to both projects!

## Step 9: Test Deployment

Once billing is enabled and all steps are complete:

```bash
git add .
git commit -m "Add NRW Web project configuration"
git push origin main
```

This will trigger automatic deployment to **BOTH** projects simultaneously!

## Workflow Behavior

**Every push to `main` will now:**
1. Deploy to Site 1 (using `cloudbuild.yaml`)
2. Deploy to Site 2 (using `cloudbuild-nrw.yaml`)
3. Both deployments run in parallel
4. GitHub Actions shows both with separate status indicators

## Adding More Projects

To add a third project, just:
1. Follow steps 1-6 above for the new project
2. Create a new `cloudbuild-newproject.yaml`
3. Add to `.github/workflows/deploy.yml` matrix:
   ```yaml
   - name: "New Project Name"
     id: newprojectid
     secret: GCP_SA_KEY_NEWPROJECT
     config: cloudbuild-newproject.yaml
   ```
4. Add `GCP_SA_KEY_NEWPROJECT` secret to GitHub

## Current Projects

- **Lucky Pup Photography**
  - Project ID: `luckypupphotography`
  - GitHub Secret: `GCP_SA_KEY_LUCKYPUP`
  - Config: `cloudbuild.yaml`
  - Auto-deploy: Yes (on push to main)

- **NRW Photography**
  - Project ID: `nrwphotography`
  - GitHub Secret: `GCP_SA_KEY_NRW`
  - Config: `cloudbuild-nrw.yaml`
  - Auto-deploy: Yes (on push to main)

---

**Need help?** Check the main [README](README.md) or [GITHUB_SETUP](GITHUB_SETUP.md).

