<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Search Articles Action
 *
 * @since  1.0.0
 */
class SearchArticlesAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Search results
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        try {
            $db = Factory::getDbo();
            if (!$db) {
                return ['error' => 'Database connection not available'];
            }
            $query = $db->getQuery(true)
                ->select('a.id, a.title, a.alias, a.catid, c.title AS category_title, a.state, a.featured, a.introtext')
                ->from($db->quoteName('#__content', 'a'))
                ->leftJoin($db->quoteName('#__categories', 'c') . ' ON a.catid = c.id');

            // Apply search term if provided - accept multiple parameter names
            $searchTerm = $parameters['search'] ?? $parameters['query'] ?? $parameters['keywords'] ?? null;
            
            // If keywords is an array, join them
            if (is_array($searchTerm)) {
                $searchTerm = implode(' ', $searchTerm);
            }
            
            if (!empty($searchTerm)) {
                $search = $db->quote('%' . $db->escape($searchTerm, true) . '%');
                $query->where('(a.title LIKE ' . $search . ' OR a.introtext LIKE ' . $search . ' OR a.fulltext LIKE ' . $search . ')');
            }

            // Filter by category
            if (!empty($parameters['category_id'])) {
                $query->where($db->quoteName('a.catid') . ' = ' . (int) $parameters['category_id']);
            }

            // Filter by state
            if (isset($parameters['state'])) {
                $query->where($db->quoteName('a.state') . ' = ' . (int) $parameters['state']);
            }

            // Filter by featured
            if (isset($parameters['featured'])) {
                $query->where($db->quoteName('a.featured') . ' = ' . (int) $parameters['featured']);
            }

            // Limit results (max 100 to prevent excessive data retrieval)
            $limit = isset($parameters['limit']) ? min((int) $parameters['limit'], 100) : 20;
            $query->setLimit($limit);

            $db->setQuery($query);
            $articles = $db->loadObjectList();

            return [
                'count' => count($articles),
                'articles' => $articles
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to search articles: ' . $e->getMessage());
        }
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Search articles by keyword, category, state, or featured status. Parameters: search (optional), category_id (optional), state (optional), featured (optional), limit (default 20).';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}

