const path = require('path');
const fs = require('fs');
const Database = require('better-sqlite3');

const dataDir = path.join(__dirname, '..', 'data');
const dbPath = path.join(dataDir, 'db.sqlite');

if (!fs.existsSync(dataDir)) {
  fs.mkdirSync(dataDir, { recursive: true });
}

const db = new Database(dbPath);
db.pragma('journal_mode = WAL');

db.exec(`
  CREATE TABLE IF NOT EXISTS albums (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    title TEXT NOT NULL,
    code TEXT NOT NULL UNIQUE,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    cover_photo_id INTEGER,
    FOREIGN KEY (cover_photo_id) REFERENCES photos(id) ON DELETE SET NULL
  );
  
  CREATE TABLE IF NOT EXISTS photos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    album_id INTEGER NOT NULL,
    filename TEXT NOT NULL,
    original_name TEXT,
    mime_type TEXT,
    size_bytes INTEGER,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    FOREIGN KEY (album_id) REFERENCES albums(id) ON DELETE CASCADE
  );
`);

function createAlbum(title, code) {
  const stmt = db.prepare(
    'INSERT INTO albums (title, code) VALUES (?, ?)' 
  );
  const info = stmt.run(title, code);
  return info.lastInsertRowid;
}

function listAlbums() {
  return db.prepare('SELECT id, title, code, created_at FROM albums ORDER BY created_at DESC').all();
}

function getAlbumById(albumId) {
  return db.prepare('SELECT * FROM albums WHERE id = ?').get(albumId);
}

function getAlbumByCode(code) {
  return db.prepare('SELECT * FROM albums WHERE code = ?').get(code);
}

function addPhoto(albumId, filename, originalName, mimeType, sizeBytes) {
  const stmt = db.prepare(
    'INSERT INTO photos (album_id, filename, original_name, mime_type, size_bytes) VALUES (?, ?, ?, ?, ?)' 
  );
  const info = stmt.run(albumId, filename, originalName, mimeType, sizeBytes);
  return info.lastInsertRowid;
}

function getPhotosByAlbum(albumId) {
  return db.prepare('SELECT * FROM photos WHERE album_id = ? ORDER BY created_at DESC').all(albumId);
}

module.exports = {
  createAlbum,
  listAlbums,
  getAlbumById,
  getAlbumByCode,
  addPhoto,
  getPhotosByAlbum,
  dbPath
};


