// Database abstraction layer - supports SQLite (local), PostgreSQL (Cloud SQL), or JSON
const USE_POSTGRES = process.env.USE_POSTGRES === 'true';
const USE_JSON = process.env.USE_JSON !== 'false'; // JSON by default for cloud

// Import the appropriate database module
let dbModule;

if (USE_POSTGRES) {
  console.log('Using PostgreSQL database (Cloud SQL)');
  dbModule = require('./db-postgres');
} else if (USE_JSON) {
  console.log('Using JSON database (lightweight, cloud-friendly)');
  dbModule = require('./db-json');
} else {
  console.log('Using SQLite database (local development)');
  
  const path = require('path');
  const fs = require('fs');
  const Database = require('better-sqlite3');

  const dataDir = path.join(__dirname, '..', 'data');
  const dbPath = path.join(dataDir, 'db.sqlite');

  if (!fs.existsSync(dataDir)) {
    fs.mkdirSync(dataDir, { recursive: true });
  }

  const db = new Database(dbPath);
  db.pragma('journal_mode = WAL');

db.exec(`
  CREATE TABLE IF NOT EXISTS album_groups (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    code TEXT NOT NULL UNIQUE,
    description TEXT,
    created_at TEXT NOT NULL DEFAULT (datetime('now'))
  );

  CREATE TABLE IF NOT EXISTS albums (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    title TEXT NOT NULL,
    code TEXT NOT NULL UNIQUE,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    cover_photo_id INTEGER,
    group_id INTEGER,
    FOREIGN KEY (cover_photo_id) REFERENCES photos(id) ON DELETE SET NULL,
    FOREIGN KEY (group_id) REFERENCES album_groups(id) ON DELETE SET NULL
  );
  
  CREATE TABLE IF NOT EXISTS photos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    album_id INTEGER NOT NULL,
    filename TEXT NOT NULL,
    original_name TEXT,
    mime_type TEXT,
    size_bytes INTEGER,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    FOREIGN KEY (album_id) REFERENCES albums(id) ON DELETE CASCADE
  );
`);

// Migration: Add group_id column to existing albums table if it doesn't exist
try {
  const tableInfo = db.prepare("PRAGMA table_info(albums)").all();
  const hasGroupId = tableInfo.some(col => col.name === 'group_id');
  
  if (!hasGroupId) {
    db.exec('ALTER TABLE albums ADD COLUMN group_id INTEGER REFERENCES album_groups(id) ON DELETE SET NULL');
    console.log('✓ Migration: Added group_id column to albums table');
  }
} catch (err) {
  console.error('Migration error:', err.message);
}

// Migration: Add is_public column to existing albums table if it doesn't exist
try {
  const tableInfo = db.prepare("PRAGMA table_info(albums)").all();
  const hasIsPublic = tableInfo.some(col => col.name === 'is_public');
  
  if (!hasIsPublic) {
    db.exec('ALTER TABLE albums ADD COLUMN is_public INTEGER NOT NULL DEFAULT 0');
    console.log('✓ Migration: Added is_public column to albums table');
  }
} catch (err) {
  console.error('Migration error:', err.message);
}

// Migration: Add hot chocolate button settings
try {
  const tableInfo = db.prepare("PRAGMA table_info(albums)").all();
  const hasHotChocEnabled = tableInfo.some(col => col.name === 'hot_choc_enabled');
  const hasHotChocUrl = tableInfo.some(col => col.name === 'hot_choc_url');
  
  if (!hasHotChocEnabled) {
    db.exec('ALTER TABLE albums ADD COLUMN hot_choc_enabled INTEGER NOT NULL DEFAULT 0');
    console.log('✓ Migration: Added hot_choc_enabled column to albums table');
  }
  
  if (!hasHotChocUrl) {
    db.exec('ALTER TABLE albums ADD COLUMN hot_choc_url TEXT');
    console.log('✓ Migration: Added hot_choc_url column to albums table');
  }
} catch (err) {
  console.error('Migration error:', err.message);
}

function createAlbum(title, code) {
  const stmt = db.prepare(
    'INSERT INTO albums (title, code) VALUES (?, ?)' 
  );
  const info = stmt.run(title, code);
  return info.lastInsertRowid;
}

function listAlbums() {
  return db.prepare('SELECT id, title, code, created_at FROM albums ORDER BY created_at DESC').all();
}

function getAlbumById(albumId) {
  return db.prepare('SELECT * FROM albums WHERE id = ?').get(albumId);
}

function getAlbumByCode(code) {
  return db.prepare('SELECT * FROM albums WHERE code = ?').get(code);
}

function updateAlbumTitle(albumId, title) {
  const stmt = db.prepare('UPDATE albums SET title = ? WHERE id = ?');
  return stmt.run(title, albumId);
}

function addPhoto(albumId, filename, originalName, mimeType, sizeBytes) {
  const stmt = db.prepare(
    'INSERT INTO photos (album_id, filename, original_name, mime_type, size_bytes) VALUES (?, ?, ?, ?, ?)' 
  );
  const info = stmt.run(albumId, filename, originalName, mimeType, sizeBytes);
  return info.lastInsertRowid;
}

function getPhotosByAlbum(albumId) {
  return db.prepare('SELECT * FROM photos WHERE album_id = ? ORDER BY created_at DESC').all(albumId);
}

// Album Group functions
function createGroup(name, code, description = null) {
  const stmt = db.prepare(
    'INSERT INTO album_groups (name, code, description) VALUES (?, ?, ?)'
  );
  const info = stmt.run(name, code, description);
  return info.lastInsertRowid;
}

function listGroups() {
  return db.prepare('SELECT id, name, code, description, created_at FROM album_groups ORDER BY created_at DESC').all();
}

function getGroupById(groupId) {
  return db.prepare('SELECT * FROM album_groups WHERE id = ?').get(groupId);
}

function getGroupByCode(code) {
  return db.prepare('SELECT * FROM album_groups WHERE code = ?').get(code);
}

function getAlbumsByGroup(groupId) {
  return db.prepare('SELECT * FROM albums WHERE group_id = ? ORDER BY created_at DESC').all(groupId);
}

function assignAlbumToGroup(albumId, groupId) {
  const stmt = db.prepare('UPDATE albums SET group_id = ? WHERE id = ?');
  return stmt.run(groupId, albumId);
}

function removeAlbumFromGroup(albumId) {
  const stmt = db.prepare('UPDATE albums SET group_id = NULL WHERE id = ?');
  return stmt.run(albumId);
}

function getGroupForAlbum(albumId) {
  return db.prepare('SELECT ag.* FROM album_groups ag JOIN albums a ON a.group_id = ag.id WHERE a.id = ?').get(albumId);
}

function listPublicAlbums() {
  return db.prepare(`
    SELECT a.*, 
           (SELECT p.filename 
            FROM photos p 
            WHERE p.album_id = a.id 
            ORDER BY p.created_at ASC 
            LIMIT 1) as cover_filename,
           (SELECT COUNT(*) FROM photos p WHERE p.album_id = a.id) as photo_count
    FROM albums a 
    WHERE a.is_public = 1 
    ORDER BY a.created_at DESC
  `).all();
}

function toggleAlbumPublic(albumId, isPublic) {
  const stmt = db.prepare('UPDATE albums SET is_public = ? WHERE id = ?');
  return stmt.run(isPublic ? 1 : 0, albumId);
}

  // Export SQLite functions wrapped in consistent interface
  dbModule = {
    createAlbum,
    listAlbums,
    getAlbumById,
    getAlbumByCode,
    updateAlbumTitle,
    addPhoto,
    getPhotosByAlbum,
    createGroup,
    listGroups,
    getGroupById,
    getGroupByCode,
    getAlbumsByGroup,
    assignAlbumToGroup,
    removeAlbumFromGroup,
    getGroupForAlbum,
    listPublicAlbums,
    toggleAlbumPublic,
    dbPath
  };
}

// Export the database module (either SQLite or PostgreSQL)
module.exports = dbModule;


