const path = require('path');
const fs = require('fs');
const Database = require('better-sqlite3');

const dataDir = path.join(__dirname, '..', 'data');
const dbPath = path.join(dataDir, 'db.sqlite');

if (!fs.existsSync(dataDir)) {
  fs.mkdirSync(dataDir, { recursive: true });
}

const db = new Database(dbPath);
db.pragma('journal_mode = WAL');

db.exec(`
  CREATE TABLE IF NOT EXISTS album_groups (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    code TEXT NOT NULL UNIQUE,
    description TEXT,
    created_at TEXT NOT NULL DEFAULT (datetime('now'))
  );

  CREATE TABLE IF NOT EXISTS albums (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    title TEXT NOT NULL,
    code TEXT NOT NULL UNIQUE,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    cover_photo_id INTEGER,
    group_id INTEGER,
    FOREIGN KEY (cover_photo_id) REFERENCES photos(id) ON DELETE SET NULL,
    FOREIGN KEY (group_id) REFERENCES album_groups(id) ON DELETE SET NULL
  );
  
  CREATE TABLE IF NOT EXISTS photos (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    album_id INTEGER NOT NULL,
    filename TEXT NOT NULL,
    original_name TEXT,
    mime_type TEXT,
    size_bytes INTEGER,
    created_at TEXT NOT NULL DEFAULT (datetime('now')),
    FOREIGN KEY (album_id) REFERENCES albums(id) ON DELETE CASCADE
  );
`);

// Migration: Add group_id column to existing albums table if it doesn't exist
try {
  const tableInfo = db.prepare("PRAGMA table_info(albums)").all();
  const hasGroupId = tableInfo.some(col => col.name === 'group_id');
  
  if (!hasGroupId) {
    db.exec('ALTER TABLE albums ADD COLUMN group_id INTEGER REFERENCES album_groups(id) ON DELETE SET NULL');
    console.log('✓ Migration: Added group_id column to albums table');
  }
} catch (err) {
  console.error('Migration error:', err.message);
}

function createAlbum(title, code) {
  const stmt = db.prepare(
    'INSERT INTO albums (title, code) VALUES (?, ?)' 
  );
  const info = stmt.run(title, code);
  return info.lastInsertRowid;
}

function listAlbums() {
  return db.prepare('SELECT id, title, code, created_at FROM albums ORDER BY created_at DESC').all();
}

function getAlbumById(albumId) {
  return db.prepare('SELECT * FROM albums WHERE id = ?').get(albumId);
}

function getAlbumByCode(code) {
  return db.prepare('SELECT * FROM albums WHERE code = ?').get(code);
}

function addPhoto(albumId, filename, originalName, mimeType, sizeBytes) {
  const stmt = db.prepare(
    'INSERT INTO photos (album_id, filename, original_name, mime_type, size_bytes) VALUES (?, ?, ?, ?, ?)' 
  );
  const info = stmt.run(albumId, filename, originalName, mimeType, sizeBytes);
  return info.lastInsertRowid;
}

function getPhotosByAlbum(albumId) {
  return db.prepare('SELECT * FROM photos WHERE album_id = ? ORDER BY created_at DESC').all(albumId);
}

// Album Group functions
function createGroup(name, code, description = null) {
  const stmt = db.prepare(
    'INSERT INTO album_groups (name, code, description) VALUES (?, ?, ?)'
  );
  const info = stmt.run(name, code, description);
  return info.lastInsertRowid;
}

function listGroups() {
  return db.prepare('SELECT id, name, code, description, created_at FROM album_groups ORDER BY created_at DESC').all();
}

function getGroupById(groupId) {
  return db.prepare('SELECT * FROM album_groups WHERE id = ?').get(groupId);
}

function getGroupByCode(code) {
  return db.prepare('SELECT * FROM album_groups WHERE code = ?').get(code);
}

function getAlbumsByGroup(groupId) {
  return db.prepare('SELECT * FROM albums WHERE group_id = ? ORDER BY created_at DESC').all(groupId);
}

function assignAlbumToGroup(albumId, groupId) {
  const stmt = db.prepare('UPDATE albums SET group_id = ? WHERE id = ?');
  return stmt.run(groupId, albumId);
}

function removeAlbumFromGroup(albumId) {
  const stmt = db.prepare('UPDATE albums SET group_id = NULL WHERE id = ?');
  return stmt.run(albumId);
}

function getGroupForAlbum(albumId) {
  return db.prepare('SELECT ag.* FROM album_groups ag JOIN albums a ON a.group_id = ag.id WHERE a.id = ?').get(albumId);
}

module.exports = {
  createAlbum,
  listAlbums,
  getAlbumById,
  getAlbumByCode,
  addPhoto,
  getPhotosByAlbum,
  createGroup,
  listGroups,
  getGroupById,
  getGroupByCode,
  getAlbumsByGroup,
  assignAlbumToGroup,
  removeAlbumFromGroup,
  getGroupForAlbum,
  dbPath
};


