<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Logger;

use Joomla\CMS\Factory;
use Joomla\CMS\Log\Log;

defined('_JEXEC') or die;

/**
 * Action Logger
 * Logs all AI assistant actions and sessions
 *
 * @since  1.0.0
 */
class ActionLogger
{
    /**
     * Database driver
     *
     * @var    \Joomla\Database\DatabaseDriver
     * @since  1.0.0
     */
    private $db;

    /**
     * Whether logging is enabled
     *
     * @var    boolean
     * @since  1.0.0
     */
    private bool $enabled;

    /**
     * Constructor
     *
     * @param   boolean  $enabled  Whether logging is enabled
     *
     * @since   1.0.0
     */
    public function __construct(bool $enabled = true)
    {
        $this->db = Factory::getContainer()->get('DatabaseDriver');
        $this->enabled = $enabled;
        $this->ensureTablesExist();
    }

    /**
     * Ensure logging tables exist
     *
     * @return  void
     * @since   1.0.0
     */
    private function ensureTablesExist(): void
    {
        try {
            // Create sessions table if not exists
            $query = "CREATE TABLE IF NOT EXISTS `#__aiassistant_sessions` (
                `id` VARCHAR(50) NOT NULL,
                `user_id` INT NOT NULL,
                `prompt` TEXT NOT NULL,
                `status` VARCHAR(50) NOT NULL,
                `result` MEDIUMTEXT NULL,
                `started_at` DATETIME NOT NULL,
                `ended_at` DATETIME NULL,
                PRIMARY KEY (`id`),
                INDEX `idx_user_started` (`user_id`, `started_at`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

            $this->db->setQuery($query);
            $this->db->execute();

            // Create actions table if not exists
            $query = "CREATE TABLE IF NOT EXISTS `#__aiassistant_actions` (
                `id` INT AUTO_INCREMENT,
                `session_id` VARCHAR(50) NOT NULL,
                `action_type` VARCHAR(100) NOT NULL,
                `parameters` TEXT NULL,
                `result` TEXT NULL,
                `status` VARCHAR(50) NOT NULL,
                `error` TEXT NULL,
                `executed_at` DATETIME NOT NULL,
                PRIMARY KEY (`id`),
                INDEX `idx_session` (`session_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;";

            $this->db->setQuery($query);
            $this->db->execute();

        } catch (\Exception $e) {
            Log::add('Failed to create AI Assistant tables: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }
    }

    /**
     * Start a new session
     *
     * @param   string  $prompt  User's prompt
     *
     * @return  string  Session ID
     * @since   1.0.0
     */
    public function startSession(string $prompt): string
    {
        $sessionId = 'aia_' . uniqid() . '_' . time();
        
        if (!$this->enabled) {
            return $sessionId;
        }

        try {
            $user = Factory::getApplication()->getIdentity();

            $query = $this->db->getQuery(true)
                ->insert($this->db->quoteName('#__aiassistant_sessions'))
                ->columns([
                    $this->db->quoteName('id'),
                    $this->db->quoteName('user_id'),
                    $this->db->quoteName('prompt'),
                    $this->db->quoteName('status'),
                    $this->db->quoteName('started_at')
                ])
                ->values(
                    $this->db->quote($sessionId) . ', ' .
                    (int) $user->id . ', ' .
                    $this->db->quote($prompt) . ', ' .
                    $this->db->quote('running') . ', ' .
                    $this->db->quote(Factory::getDate()->toSql())
                );

            $this->db->setQuery($query);
            $this->db->execute();

        } catch (\Exception $e) {
            Log::add('Failed to start session: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }

        return $sessionId;
    }

    /**
     * End a session
     *
     * @param   string  $sessionId  Session ID
     * @param   array   $result     Session result
     *
     * @return  void
     * @since   1.0.0
     */
    public function endSession(string $sessionId, array $result): void
    {
        if (!$this->enabled) {
            return;
        }

        try {
            $query = $this->db->getQuery(true)
                ->update($this->db->quoteName('#__aiassistant_sessions'))
                ->set([
                    $this->db->quoteName('status') . ' = ' . $this->db->quote($result['status'] ?? 'completed'),
                    $this->db->quoteName('result') . ' = ' . $this->db->quote(json_encode($result)),
                    $this->db->quoteName('ended_at') . ' = ' . $this->db->quote(Factory::getDate()->toSql())
                ])
                ->where($this->db->quoteName('id') . ' = ' . $this->db->quote($sessionId));

            $this->db->setQuery($query);
            $this->db->execute();

        } catch (\Exception $e) {
            Log::add('Failed to end session: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }
    }

    /**
     * Log an action
     *
     * @param   string  $sessionId  Session ID
     * @param   array   $action     Action data
     *
     * @return  void
     * @since   1.0.0
     */
    public function logAction(string $sessionId, array $action): void
    {
        if (!$this->enabled) {
            return;
        }

        try {
            $query = $this->db->getQuery(true)
                ->insert($this->db->quoteName('#__aiassistant_actions'))
                ->columns([
                    $this->db->quoteName('session_id'),
                    $this->db->quoteName('action_type'),
                    $this->db->quoteName('parameters'),
                    $this->db->quoteName('result'),
                    $this->db->quoteName('status'),
                    $this->db->quoteName('error'),
                    $this->db->quoteName('executed_at')
                ])
                ->values(
                    $this->db->quote($sessionId) . ', ' .
                    $this->db->quote($action['type'] ?? 'unknown') . ', ' .
                    $this->db->quote(json_encode($action['parameters'] ?? [])) . ', ' .
                    $this->db->quote(json_encode($action['result'] ?? null)) . ', ' .
                    $this->db->quote($action['status'] ?? 'unknown') . ', ' .
                    $this->db->quote($action['error'] ?? null) . ', ' .
                    $this->db->quote(Factory::getDate()->toSql())
                );

            $this->db->setQuery($query);
            $this->db->execute();

        } catch (\Exception $e) {
            Log::add('Failed to log action: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }
    }

    /**
     * Log an error
     *
     * @param   string  $sessionId  Session ID
     * @param   array   $error      Error data
     *
     * @return  void
     * @since   1.0.0
     */
    public function logError(string $sessionId, array $error): void
    {
        if (!$this->enabled) {
            return;
        }

        try {
            $query = $this->db->getQuery(true)
                ->update($this->db->quoteName('#__aiassistant_sessions'))
                ->set([
                    $this->db->quoteName('status') . ' = ' . $this->db->quote('error'),
                    $this->db->quoteName('result') . ' = ' . $this->db->quote(json_encode($error)),
                    $this->db->quoteName('ended_at') . ' = ' . $this->db->quote(Factory::getDate()->toSql())
                ])
                ->where($this->db->quoteName('id') . ' = ' . $this->db->quote($sessionId));

            $this->db->setQuery($query);
            $this->db->execute();

            Log::add('AI Assistant error in session ' . $sessionId . ': ' . ($error['error'] ?? 'Unknown error'), Log::ERROR, 'aiassistant');

        } catch (\Exception $e) {
            Log::add('Failed to log error: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }
    }

    /**
     * Get session actions
     *
     * @param   string  $sessionId  Session ID
     *
     * @return  array   Actions
     * @since   1.0.0
     */
    public function getSessionActions(string $sessionId): array
    {
        try {
            $query = $this->db->getQuery(true)
                ->select('*')
                ->from($this->db->quoteName('#__aiassistant_actions'))
                ->where($this->db->quoteName('session_id') . ' = ' . $this->db->quote($sessionId))
                ->order($this->db->quoteName('id'));

            $this->db->setQuery($query);
            $actions = $this->db->loadAssocList();

            // Decode JSON fields
            foreach ($actions as &$action) {
                $action['parameters'] = json_decode($action['parameters'], true);
                $action['result'] = json_decode($action['result'], true);
            }

            return $actions;

        } catch (\Exception $e) {
            Log::add('Failed to get session actions: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
            return [];
        }
    }

    /**
     * Update an action
     *
     * @param   int    $actionId  Action ID
     * @param   array  $data      Update data
     *
     * @return  void
     * @since   1.0.0
     */
    public function updateAction(int $actionId, array $data): void
    {
        if (!$this->enabled) {
            return;
        }

        try {
            $query = $this->db->getQuery(true)
                ->update($this->db->quoteName('#__aiassistant_actions'))
                ->set([
                    $this->db->quoteName('status') . ' = ' . $this->db->quote($data['status'] ?? 'unknown'),
                    $this->db->quoteName('result') . ' = ' . $this->db->quote(json_encode($data['result'] ?? null)),
                    $this->db->quoteName('error') . ' = ' . $this->db->quote($data['error'] ?? null)
                ])
                ->where($this->db->quoteName('id') . ' = ' . (int) $actionId);

            $this->db->setQuery($query);
            $this->db->execute();

        } catch (\Exception $e) {
            Log::add('Failed to update action: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
        }
    }

    /**
     * Get recent sessions for current user
     *
     * @param   int  $limit  Number of sessions to retrieve
     *
     * @return  array
     * @since   1.0.0
     */
    public function getRecentSessions(int $limit = 10): array
    {
        try {
            $user = Factory::getApplication()->getIdentity();

            $query = $this->db->getQuery(true)
                ->select('*')
                ->from($this->db->quoteName('#__aiassistant_sessions'))
                ->where($this->db->quoteName('user_id') . ' = ' . (int) $user->id)
                ->order($this->db->quoteName('started_at') . ' DESC')
                ->setLimit($limit);

            $this->db->setQuery($query);
            $sessions = $this->db->loadAssocList();

            // Decode result JSON
            foreach ($sessions as &$session) {
                if ($session['result']) {
                    $session['result'] = json_decode($session['result'], true);
                }
            }

            return $sessions;

        } catch (\Exception $e) {
            Log::add('Failed to get recent sessions: ' . $e->getMessage(), Log::ERROR, 'aiassistant');
            return [];
        }
    }
}

