<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Get Site Info Action
 *
 * @since  1.0.0
 */
class GetSiteInfoAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Site information
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        $app = Factory::getApplication();
        $config = $app->getConfig();
        $db = Factory::getContainer()->get('DatabaseDriver');
        
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }

        // Get template info
        $query = $db->getQuery(true)
            ->select('template, home')
            ->from($db->quoteName('#__template_styles'))
            ->where($db->quoteName('client_id') . ' = 0')
            ->where($db->quoteName('home') . ' = 1');
        
        $db->setQuery($query);
        $template = $db->loadObject();

        // Get installed extensions count
        $query = $db->getQuery(true)
            ->select('COUNT(*)')
            ->from($db->quoteName('#__extensions'));
        
        $db->setQuery($query);
        $extensionsCount = $db->loadResult();

        return [
            'site_name' => $config ? $config->get('sitename') : 'Unknown',
            'site_url' => \Joomla\CMS\Uri\Uri::root(),
            'joomla_version' => JVERSION,
            'php_version' => PHP_VERSION,
            'database_type' => $config ? $config->get('dbtype') : 'Unknown',
            'active_template' => $template->template ?? 'unknown',
            'site_offline' => $config ? (bool) $config->get('offline') : false,
            'timezone' => $config ? $config->get('offset', 'UTC') : 'UTC',
            'language' => $app->getLanguage()->getTag(),
            'extensions_count' => $extensionsCount,
            'debug_mode' => $config ? (bool) $config->get('debug') : false
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Get general site information including name, URL, Joomla version, template, and configuration.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}

