<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;
use Joomla\Registry\Registry;

defined('_JEXEC') or die;

/**
 * Read Module Action
 *
 * @since  1.0.0
 */
class ReadModuleAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Module data
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for ReadModuleAction');
        }

            $db = Factory::getDbo();
        if (!$db) {
            return ['error' => 'Database connection not available'];
        }
        $moduleId = $parameters['module_id'];

        $query = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__modules'))
            ->where($db->quoteName('id') . ' = ' . (int) $moduleId);

        $db->setQuery($query);
        $module = $db->loadObject();

        if (!$module) {
            throw new \RuntimeException("Module with ID {$moduleId} not found");
        }

        // Parse parameters
        $params = new Registry($module->params);

        return [
            'id' => $module->id,
            'title' => $module->title,
            'module' => $module->module,
            'position' => $module->position,
            'content' => $module->content,
            'published' => $module->published,
            'params' => $params->toArray(),
            'ordering' => $module->ordering,
            'language' => $module->language,
            'note' => $module->note
        ];
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Read a module by ID. Returns module data including title, content, position, parameters, etc.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['module_id'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return isset($parameters['module_id']) && is_numeric($parameters['module_id']);
    }
}

