<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * List Modules Action
 *
 * @since  1.0.0
 */
class ListModulesAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Modules list
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        try {
            $db = Factory::getContainer()->get('DatabaseDriver');
            if (!$db) {
                return ['error' => 'Database connection not available'];
            }
            $query = $db->getQuery(true)
                ->select('id, title, module, position, published, language')
                ->from($db->quoteName('#__modules'));

            // Filter by position
            if (!empty($parameters['position'])) {
                $query->where($db->quoteName('position') . ' = ' . $db->quote($parameters['position']));
            }

            // Filter by module type
            if (!empty($parameters['module_type'])) {
                $query->where($db->quoteName('module') . ' = ' . $db->quote($parameters['module_type']));
            }

            // Filter by published state
            if (isset($parameters['published'])) {
                $query->where($db->quoteName('published') . ' = ' . (int) $parameters['published']);
            }

            $query->order($db->quoteName('position') . ', ' . $db->quoteName('ordering'));

            $db->setQuery($query);
            $modules = $db->loadObjectList();

            return [
                'count' => count($modules),
                'modules' => $modules
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to list modules: ' . $e->getMessage());
        }
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'List all modules. Optional filters: position, module_type, published.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return [];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return true;
    }
}

