# Usage Limits & Cost Control Guide

This guide explains how to implement and manage usage limits for your NRW Web SaaS product to prevent unexpected costs.

## 📊 Tier System Overview

### Implemented Tiers

| Tier | Photos | Storage | Bandwidth/month | Albums | Max File Size | Monthly Cost (You) | Suggested Price |
|------|--------|---------|-----------------|--------|---------------|-------------------|-----------------|
| **Free**  | 10 | 100 MB | 1 GB | 1 | 5 MB | £0 | Free |
| **Starter** | 100 | 2 GB | 5 GB | 1 | 10 MB | £0.04-0.60 | £5/month |
| **Pro** | 500 | 10 GB | 50 GB | 10 | 20 MB | £0.20-1.50 | £15/month |
| **Business** | 5,000 | 100 GB | 500 GB | Unlimited | 50 MB | £2-6 | £50/month |
| **Enterprise** | 99,999 | 1 TB | 5 TB | Unlimited | 100 MB | Custom | Custom |

## 🛠️ How to Set Up

### 1. Set Tier for Each Customer Deployment

When deploying a customer site, set the `SUBSCRIPTION_TIER` environment variable:

```bash
# Deploy with Starter tier (£10/month)
gcloud run deploy customer-site \
  --source . \
  --set-env-vars="SUBSCRIPTION_TIER=starter" \
  --region europe-west1 \
  --project=your-project

# Deploy with Professional tier (£25/month)
gcloud run deploy customer-site-pro \
  --source . \
  --set-env-vars="SUBSCRIPTION_TIER=professional" \
  --region europe-west1 \
  --project=your-project

# Deploy with Business tier (£75/month)
gcloud run deploy customer-site-business \
  --source . \
  --set-env-vars="SUBSCRIPTION_TIER=business" \
  --region europe-west1 \
  --project=your-project
```

### 2. Limits Are Enforced Automatically

The system will:
- ✅ Block uploads exceeding photo count limits
- ✅ Block uploads exceeding storage limits
- ✅ Block individual files exceeding max file size
- ✅ Block album creation exceeding album limits
- ✅ Show clear error messages to users
- ✅ Display usage stats in admin dashboard

### 3. Usage Dashboard

Admins can see their current usage at `/admin`:
- Total photos uploaded
- Storage used (with progress bar)
- Albums created
- Current tier limits

## 💰 Google Cloud Budget Alerts

Set up budget alerts to get notified before costs exceed limits:

### Per-Customer Budget Alerts

For each customer project:

```bash
# Set project
gcloud config set project customer-project-id

# Create budget (£10/month example for Starter tier customers)
gcloud billing budgets create \
  --billing-account=YOUR-BILLING-ACCOUNT-ID \
  --display-name="Customer Starter Tier Budget" \
  --budget-amount=10GBP \
  --threshold-rule=percent=50 \
  --threshold-rule=percent=90 \
  --threshold-rule=percent=100
```

You'll receive email alerts at:
- 50% of budget (£5) - Warning
- 90% of budget (£9) - High warning
- 100% of budget (£10) - Over budget

### Master Budget Alert

Set an overall budget across all customers:

```bash
gcloud billing budgets create \
  --billing-account=YOUR-BILLING-ACCOUNT-ID \
  --display-name="NRW Web Total Monthly Budget" \
  --budget-amount=500GBP \
  --threshold-rule=percent=75 \
  --threshold-rule=percent=90 \
  --threshold-rule=percent=100 \
  --all-updates-rule-pubsub-topic=projects/YOUR-PROJECT/topics/budget-alerts
```

## 🚨 Cost Protection Features

### 1. File Size Limits (Already Implemented)
- Multer configured with `fileSize: 50MB` max
- Per-tier limits enforce smaller sizes for lower tiers
- Files rejected before upload completes

### 2. Storage Quotas (Already Implemented)
- Photo count tracked in database
- Storage bytes summed from all photos
- Upload blocked if limits exceeded

### 3. Cloud Storage Bucket Quotas (Recommended)

Set quota on each customer's bucket to hard-cap storage:

```bash
# Get bucket name for customer
BUCKET_NAME="customer-bucket-name"

# For Starter tier (2GB)
gcloud storage buckets update gs://${BUCKET_NAME} \
  --soft-delete-duration=0s

# Note: GCS doesn't have direct size quotas, but you can:
# 1. Monitor via Cloud Monitoring
# 2. Set up alerts when bucket size exceeds threshold
# 3. Use your app limits (already implemented)
```

### 4. Cloud Run Memory/CPU Limits

Prevent runaway costs from traffic spikes:

```bash
gcloud run services update customer-site \
  --memory=512Mi \           # Sufficient for image processing
  --cpu=1 \                  # Single CPU core
  --max-instances=10 \       # Limit scaling
  --concurrency=80 \         # Requests per instance
  --region=europe-west1
```

**Cost impact:**
- 10 instances × 512MB × 80 concurrent = Max £2-3/hour during spike
- Auto-scales down to 0 when idle = £0

## 📈 Monitoring Costs

### Real-Time Cost Monitoring

Set up a monitoring dashboard:

1. Go to Google Cloud Console → Monitoring → Dashboards
2. Create dashboard with widgets for:
   - **Cloud Storage**: Bucket size per customer
   - **Cloud Run**: Request count, instance count
   - **Network Egress**: Bandwidth usage
   - **Billing**: Daily spend

### Recommended Alerts

```bash
# Alert when bucket exceeds tier limit
gcloud alpha monitoring policies create \
  --notification-channels=CHANNEL-ID \
  --display-name="Customer Bucket Size Alert" \
  --condition-display-name="Bucket over 2GB" \
  --condition-threshold-value=2147483648 \  # 2GB in bytes
  --condition-threshold-duration=300s \
  --condition-filter='resource.type="gcs_bucket" AND metric.type="storage.googleapis.com/storage/total_bytes"'
```

## 🔐 Additional Protection Measures

### 1. Rate Limiting (Already Implemented)
```javascript
// Already in server.js
const uploadLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 500, // 500 uploads per hour
});
```

### 2. Prevent Abuse

Add IP-based restrictions if needed:

```javascript
// In server.js
const extremeAbuseLimiter = rateLimit({
  windowMs: 24 * 60 * 60 * 1000, // 24 hours
  max: 1000, // 1000 uploads per day per IP
  message: 'Too many uploads from this IP, please try again tomorrow.'
});
```

### 3. Implement Soft Deletion

Instead of immediate deletion, mark photos as deleted and clean up later:

```javascript
// Gives you time to recover costs if customer disputes
async function softDeletePhoto(photoId) {
  // Mark as deleted, clean up after 30 days
  await db.updatePhoto(photoId, { deleted_at: new Date() });
}
```

## 📋 Cost Calculation Examples

### Starter Tier Customer (£5/month)
```
Storage: 2GB × £0.020 = £0.04/month
Bandwidth: 5GB × £0.12 = £0.60/month
Cloud Run: Free tier (2M requests)
Total: ~£0.64/month
Profit: £5 - £0.64 = £4.36/month (87% margin)
```

### Pro Tier Customer (£15/month)
```
Storage: 10GB × £0.020 = £0.20/month
Bandwidth: 50GB × £0.12 = £6.00/month
Cloud Run: Free tier likely sufficient
Total: ~£6.20/month (if they use all bandwidth)
Profit: £15 - £6.20 = £8.80/month (59% margin)
```

### Business Tier Customer (£50/month)
```
Storage: 100GB × £0.020 = £2.00/month
Bandwidth: 500GB × £0.12 = £60.00/month (ALERT!)
Cloud Run: £5/month (high traffic)
Total: ~£67/month if maxed out ⚠️
```

**Note:** Business tier needs monitoring - if customer uses full 500GB bandwidth, you lose money. Consider:
- Setting bandwidth at 100GB for £50/month
- Or charge £100/month for 500GB tier

## 🎯 Recommended Actions

1. **Start Conservative**: Begin with lower limits, easier to increase than decrease
2. **Monitor First Month**: Track actual costs vs revenue for each tier
3. **Adjust Pricing**: If costs approach 50% of revenue, increase prices or decrease limits
4. **Add Overage Charges**: Charge £0.15/GB for bandwidth over limit
5. **Annual Discounts**: Offer 2 months free for annual payment (improves cash flow)

## 🚀 Deployment Checklist

For each new customer:
- [ ] Set `SUBSCRIPTION_TIER` environment variable
- [ ] Create per-customer budget alert (2x their payment amount)
- [ ] Set Cloud Run max instances (10 for starter/pro, 50 for business)
- [ ] Monitor costs for first week
- [ ] Send welcome email with tier limits

## 📞 When Customer Wants to Upgrade

```bash
# Update tier
gcloud run services update customer-site \
  --update-env-vars=SUBSCRIPTION_TIER=pro \
  --region=europe-west1

# Update budget alert
gcloud billing budgets update BUDGET-ID \
  --budget-amount=30GBP
```

## 💡 Pro Tips

1. **Bundle Storage & Bandwidth**: Customers don't understand bandwidth limits well. Market as "5,000 photos with unlimited views" instead of "10GB storage + 50GB bandwidth"

2. **Overage Forgiveness**: First time customer hits limit, send friendly email offering one-time forgiveness if they upgrade

3. **Usage Reports**: Send monthly emails showing usage stats - encourages upgrades before limits hit

4. **Seasonal Pricing**: Offer deals in January (wedding photographers planning season)

---

## Support

Questions? Check the main README or contact support.

