const fs = require('fs').promises;
const fsSync = require('fs');
const path = require('path');
const { Storage } = require('@google-cloud/storage');

const USE_CLOUD_STORAGE = process.env.USE_CLOUD_STORAGE === 'true';
const GCS_BUCKET_NAME = process.env.GCS_BUCKET_NAME;
const DATA_FILE = 'data.json';

// Initialize Cloud Storage client
const storage = USE_CLOUD_STORAGE ? new Storage() : null;

// In-memory cache
let dataCache = null;
let lastSaved = Date.now();
const SAVE_INTERVAL = 5000; // Save every 5 seconds if changes

// Default data structure
const defaultData = {
  albums: [],
  groups: [],
  photos: [],
  nextId: {
    albums: 1,
    groups: 1,
    photos: 1
  }
};

/**
 * Load data from local file or Cloud Storage
 */
async function loadData() {
  if (dataCache) {
    return dataCache;
  }

  try {
    let dataJson;
    
    if (USE_CLOUD_STORAGE) {
      // Load from Cloud Storage
      const bucket = storage.bucket(GCS_BUCKET_NAME);
      const file = bucket.file(DATA_FILE);
      
      const [exists] = await file.exists();
      if (exists) {
        const [contents] = await file.download();
        dataJson = contents.toString();
      } else {
        // Initialize with default data
        dataJson = JSON.stringify(defaultData, null, 2);
        await file.save(dataJson, {
          metadata: { contentType: 'application/json' }
        });
      }
    } else {
      // Load from local file
      const localPath = path.join(__dirname, '..', 'data', DATA_FILE);
      const dataDir = path.dirname(localPath);
      
      if (!fsSync.existsSync(dataDir)) {
        fsSync.mkdirSync(dataDir, { recursive: true });
      }
      
      if (fsSync.existsSync(localPath)) {
        dataJson = await fs.readFile(localPath, 'utf-8');
      } else {
        dataJson = JSON.stringify(defaultData, null, 2);
        await fs.writeFile(localPath, dataJson, 'utf-8');
      }
    }
    
    dataCache = JSON.parse(dataJson);
    console.log('✓ JSON database loaded');
    return dataCache;
  } catch (error) {
    console.error('Error loading data:', error);
    dataCache = { ...defaultData };
    return dataCache;
  }
}

/**
 * Save data to local file or Cloud Storage
 */
async function saveData(immediate = false) {
  if (!dataCache) return;
  
  const now = Date.now();
  if (!immediate && now - lastSaved < SAVE_INTERVAL) {
    // Debounce saves
    return;
  }
  
  try {
    const dataJson = JSON.stringify(dataCache, null, 2);
    
    if (USE_CLOUD_STORAGE) {
      const bucket = storage.bucket(GCS_BUCKET_NAME);
      const file = bucket.file(DATA_FILE);
      await file.save(dataJson, {
        metadata: { contentType: 'application/json' }
      });
    } else {
      const localPath = path.join(__dirname, '..', 'data', DATA_FILE);
      await fs.writeFile(localPath, dataJson, 'utf-8');
    }
    
    lastSaved = now;
  } catch (error) {
    console.error('Error saving data:', error);
    throw error;
  }
}

// Auto-save on exit
process.on('SIGTERM', async () => {
  await saveData(true);
  process.exit(0);
});

process.on('SIGINT', async () => {
  await saveData(true);
  process.exit(0);
});

// Albums
async function createAlbum(title, code) {
  const data = await loadData();
  const id = data.nextId.albums++;
  
  const album = {
    id,
    title,
    code,
    created_at: new Date().toISOString(),
    cover_photo_id: null,
    group_id: null,
    is_public: 0,
    sort_order: id // Default sort order is the ID
  };
  
  data.albums.push(album);
  await saveData();
  return id;
}

async function listAlbums() {
  const data = await loadData();
  return data.albums
    .slice()
    .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
}

async function getAlbumById(albumId) {
  const data = await loadData();
  return data.albums.find(a => a.id === albumId) || null;
}

async function getAlbumByCode(code) {
  const data = await loadData();
  return data.albums.find(a => a.code.toUpperCase() === code.toUpperCase()) || null;
}

async function updateAlbumTitle(albumId, title) {
  const data = await loadData();
  const album = data.albums.find(a => a.id === albumId);
  if (album) {
    album.title = title;
    await saveData();
  }
}

async function toggleAlbumPublic(albumId, isPublic) {
  const data = await loadData();
  const album = data.albums.find(a => a.id === albumId);
  if (album) {
    album.is_public = isPublic ? 1 : 0;
    await saveData();
  }
}

async function deleteAlbum(albumId) {
  const data = await loadData();
  // Remove the album
  data.albums = data.albums.filter(a => a.id !== albumId);
  // Remove all photos in the album
  data.photos = data.photos.filter(p => p.album_id !== albumId);
  await saveData(true);
}

// Photos
async function addPhoto(albumId, filename, originalName, mimeType, sizeBytes) {
  const data = await loadData();
  const id = data.nextId.photos++;
  
  const photo = {
    id,
    album_id: albumId,
    filename,
    original_name: originalName,
    mime_type: mimeType,
    size_bytes: sizeBytes,
    created_at: new Date().toISOString()
  };
  
  data.photos.push(photo);
  await saveData();
  return id;
}

async function getPhotosByAlbum(albumId) {
  const data = await loadData();
  return data.photos
    .filter(p => p.album_id === albumId)
    .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
}

// Groups
async function createGroup(name, code, description = null) {
  const data = await loadData();
  const id = data.nextId.groups++;
  
  const group = {
    id,
    name,
    code,
    description,
    created_at: new Date().toISOString()
  };
  
  data.groups.push(group);
  await saveData();
  return id;
}

async function listGroups() {
  const data = await loadData();
  return data.groups
    .slice()
    .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
}

async function getGroupById(groupId) {
  const data = await loadData();
  return data.groups.find(g => g.id === groupId) || null;
}

async function getGroupByCode(code) {
  const data = await loadData();
  return data.groups.find(g => g.code.toUpperCase() === code.toUpperCase()) || null;
}

async function getAlbumsByGroup(groupId) {
  const data = await loadData();
  return data.albums
    .filter(a => a.group_id === groupId)
    .sort((a, b) => new Date(b.created_at) - new Date(a.created_at));
}

async function assignAlbumToGroup(albumId, groupId) {
  const data = await loadData();
  const album = data.albums.find(a => a.id === albumId);
  if (album) {
    album.group_id = groupId;
    await saveData();
  }
}

async function removeAlbumFromGroup(albumId) {
  const data = await loadData();
  const album = data.albums.find(a => a.id === albumId);
  if (album) {
    album.group_id = null;
    await saveData();
  }
}

async function deleteGroup(groupId, deleteAlbumsInGroup = false) {
  const data = await loadData();
  
  if (deleteAlbumsInGroup) {
    // Delete all albums in this group (and their photos)
    const albumsInGroup = data.albums.filter(a => a.group_id === groupId);
    for (const album of albumsInGroup) {
      data.albums = data.albums.filter(a => a.id !== album.id);
      data.photos = data.photos.filter(p => p.album_id !== album.id);
    }
  } else {
    // Just remove group association from albums
    data.albums.forEach(album => {
      if (album.group_id === groupId) {
        album.group_id = null;
      }
    });
  }
  
  // Remove the group
  data.groups = data.groups.filter(g => g.id !== groupId);
  await saveData(true);
}

async function deletePhoto(photoId) {
  const data = await loadData();
  data.photos = data.photos.filter(p => p.id !== photoId);
  await saveData(true);
}

async function getGroupForAlbum(albumId) {
  const data = await loadData();
  const album = data.albums.find(a => a.id === albumId);
  if (!album || !album.group_id) return null;
  return data.groups.find(g => g.id === album.group_id) || null;
}

async function listPublicAlbums() {
  const data = await loadData();
  const publicAlbums = data.albums.filter(a => a.is_public === 1);
  
  // Add cover photo and photo count
  return publicAlbums.map(album => {
    const photos = data.photos.filter(p => p.album_id === album.id);
    const coverPhoto = photos.sort((a, b) => 
      new Date(a.created_at) - new Date(b.created_at)
    )[0];
    
    return {
      ...album,
      cover_filename: coverPhoto ? coverPhoto.filename : null,
      photo_count: photos.length
    };
  }).sort((a, b) => (a.sort_order || a.id) - (b.sort_order || b.id));
}

async function updateAlbumSortOrder(albumIds) {
  const data = await loadData();
  // albumIds is an array of album IDs in the desired order
  albumIds.forEach((albumId, index) => {
    const album = data.albums.find(a => a.id === albumId);
    if (album) {
      album.sort_order = index;
    }
  });
  await saveData(true);
}

module.exports = {
  createAlbum,
  listAlbums,
  getAlbumById,
  getAlbumByCode,
  updateAlbumTitle,
  addPhoto,
  getPhotosByAlbum,
  createGroup,
  listGroups,
  getGroupById,
  getGroupByCode,
  getAlbumsByGroup,
  assignAlbumToGroup,
  removeAlbumFromGroup,
  getGroupForAlbum,
  listPublicAlbums,
  toggleAlbumPublic,
  deleteAlbum,
  deleteGroup,
  deletePhoto,
  updateAlbumSortOrder,
};

