#!/bin/bash
# Deployment script for NRW Web to Google Cloud Run
# This script automates the deployment process

set -e  # Exit on error

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}Lucky Pup Photography - Cloud Run Deployment${NC}"
echo "================================================"
echo ""

# Check if gcloud is installed
if ! command -v gcloud &> /dev/null; then
    echo -e "${RED}Error: gcloud CLI is not installed${NC}"
    echo "Please install it from: https://cloud.google.com/sdk/docs/install"
    exit 1
fi

# Get project ID
PROJECT_ID=$(gcloud config get-value project 2>/dev/null)
if [ -z "$PROJECT_ID" ]; then
    echo -e "${RED}Error: No GCP project configured${NC}"
    echo "Run: gcloud config set project YOUR_PROJECT_ID"
    exit 1
fi

echo -e "${GREEN}✓${NC} Using GCP Project: ${PROJECT_ID}"

# Set default values
REGION=${REGION:-us-central1}
SERVICE_NAME=${SERVICE_NAME:-photowebsite}
BUCKET_NAME=${BUCKET_NAME:-${PROJECT_ID}-photowebsite}

echo ""
echo "Configuration:"
echo "  Region: $REGION"
echo "  Service Name: $SERVICE_NAME"
echo "  Bucket Name: $BUCKET_NAME"
echo "  Database: JSON (no Cloud SQL - FREE!)"
echo ""

# Ask for confirmation
read -p "Continue with deployment? (y/n) " -n 1 -r
echo ""
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Deployment cancelled"
    exit 0
fi

echo ""
echo -e "${YELLOW}Step 1: Enabling Required APIs...${NC}"
gcloud services enable run.googleapis.com \
  cloudbuild.googleapis.com \
  sqladmin.googleapis.com \
  secretmanager.googleapis.com \
  containerregistry.googleapis.com \
  storage.googleapis.com
echo -e "${GREEN}✓${NC} APIs enabled"

echo ""
echo -e "${GREEN}✓${NC} Using JSON database (no Cloud SQL needed - FREE!)"

echo ""
echo -e "${YELLOW}Step 2: Checking Cloud Storage Bucket...${NC}"
if gsutil ls gs://$BUCKET_NAME &> /dev/null; then
    echo -e "${GREEN}✓${NC} Storage bucket exists"
else
    echo "Creating Cloud Storage bucket..."
    gsutil mb -l $REGION gs://$BUCKET_NAME
    gsutil iam ch allUsers:objectViewer gs://$BUCKET_NAME
    gsutil uniformbucketlevelaccess set on gs://$BUCKET_NAME
    echo -e "${GREEN}✓${NC} Storage bucket created"
fi

echo ""
echo -e "${YELLOW}Step 3: Checking Secrets...${NC}"
PROJECT_NUMBER=$(gcloud projects describe $PROJECT_ID --format="value(projectNumber)")

# Check admin password secret
if gcloud secrets describe photowebsite-admin-password &> /dev/null; then
    echo -e "${GREEN}✓${NC} Admin password secret exists"
else
    echo "Creating admin password secret..."
    read -sp "Enter admin password: " ADMIN_PASSWORD
    echo ""
    echo -n "$ADMIN_PASSWORD" | gcloud secrets create photowebsite-admin-password \
      --data-file=- \
      --replication-policy="automatic"
    
    gcloud secrets add-iam-policy-binding photowebsite-admin-password \
      --member="serviceAccount:${PROJECT_NUMBER}-compute@developer.gserviceaccount.com" \
      --role="roles/secretmanager.secretAccessor"
    echo -e "${GREEN}✓${NC} Admin password secret created"
fi

echo ""
echo -e "${YELLOW}Step 4: Building and Deploying...${NC}"
echo "Using Cloud Build to build and deploy the application..."

gcloud builds submit --config=cloudbuild.yaml \
  --substitutions=_DEPLOY_REGION=$REGION,_GCS_BUCKET_NAME=$BUCKET_NAME

echo ""
echo -e "${GREEN}✓ Deployment Complete!${NC}"
echo ""

# Get service URL
SERVICE_URL=$(gcloud run services describe $SERVICE_NAME --region=$REGION --format='value(status.url)')
echo "Your application is now live at:"
echo -e "${GREEN}${SERVICE_URL}${NC}"
echo ""
echo "Admin panel:"
echo -e "${GREEN}${SERVICE_URL}/admin${NC}"
echo ""
echo "Next steps:"
echo "  - Visit your application"
echo "  - Create albums in the admin panel"
echo "  - Upload photos"
echo "  - Share access codes with clients"
echo ""
echo "To view logs:"
echo "  gcloud run services logs read $SERVICE_NAME --region=$REGION"
echo ""

