<?php
/**
 * @package     Joomla.Plugin
 * @subpackage  System.aiassistant
 *
 * @copyright   Copyright (C) 2025 Open Source Matters. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

namespace Joomla\Plugin\System\AiAssistant\Actions\Actions;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Plugin\System\AiAssistant\Actions\ActionInterface;

defined('_JEXEC') or die;

/**
 * Create Article Action
 *
 * @since  1.0.0
 */
class CreateArticleAction implements ActionInterface
{
    /**
     * Execute the action
     *
     * @param   array  $parameters  Action parameters
     *
     * @return  array  Creation result
     * @since   1.0.0
     */
    public function execute(array $parameters): array
    {
        if (!$this->validate($parameters)) {
            throw new \InvalidArgumentException('Invalid parameters for CreateArticleAction');
        }

        try {
            // Load the article model
            BaseDatabaseModel::addIncludePath(JPATH_ADMINISTRATOR . '/components/com_content/models');
            $model = BaseDatabaseModel::getInstance('Article', 'ContentModel');

            if (!$model) {
                throw new \RuntimeException('Failed to load article model');
            }

            // Prepare article data with defaults
            $data = [
                'title' => $parameters['title'],
                'alias' => $parameters['alias'] ?? '',
                'introtext' => $parameters['introtext'] ?? '',
                'fulltext' => $parameters['fulltext'] ?? '',
                'catid' => $parameters['catid'] ?? 2, // Uncategorised
                'state' => $parameters['state'] ?? 0, // Unpublished by default
                'language' => $parameters['language'] ?? '*',
                'access' => $parameters['access'] ?? 1, // Public
                'metadesc' => $parameters['metadesc'] ?? '',
                'metakey' => $parameters['metakey'] ?? '',
                'featured' => $parameters['featured'] ?? 0
            ];

            // Save the article
            if (!$model->save($data)) {
                throw new \RuntimeException('Failed to create article: ' . $model->getError());
            }

            $articleId = $model->getState('article.id');

            return [
                'success' => true,
                'article_id' => $articleId,
                'title' => $data['title'],
                'message' => 'Article created successfully',
                'before_state' => ['article_id' => $articleId] // Store ID for deletion on undo
            ];

        } catch (\Exception $e) {
            throw new \RuntimeException('Failed to create article: ' . $e->getMessage());
        }
    }

    /**
     * Get action description for AI
     *
     * @return  string
     * @since   1.0.0
     */
    public static function getDescription(): string
    {
        return 'Create a new article. Required: title. Optional: alias, introtext, fulltext, catid, state, language, access, metadesc, metakey, featured.';
    }

    /**
     * Get required parameters
     *
     * @return  array
     * @since   1.0.0
     */
    public static function getRequiredParameters(): array
    {
        return ['title'];
    }

    /**
     * Validate parameters
     *
     * @param   array  $parameters  Parameters to validate
     *
     * @return  boolean
     * @since   1.0.0
     */
    public function validate(array $parameters): bool
    {
        return !empty($parameters['title']);
    }
}

